from gpaw import GPAW, PW, FermiDirac
from ase import Atoms
from ase.build import fcc111, add_adsorbate
from ase.optimize import QuasiNewton, BFGS
from ase.constraints import FixAtoms
import ase.visualize
import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import curve_fit


# define function that returns a Pt(111) slab with adsorbates on top.
def build_slab_with_adsorbate(N=0, Nx=2, Ny=2, l=3):
    """ returns a Pt(111) slab with adsorbates on top.
    Input:
    - N: number of adsorbates
    - Nx: number of surface atoms in first lattice direction
    - Ny: number of surface atoms in second lattice direction
    - l: number of layers
    Output:
    - ase geometry
    """
    #construct slab and center it
    slab = fcc111('Pt', (Nx,Ny,l), vacuum=3.5)
    slab.center(axis=2)

    #add adsorbates
    for i_ads in range(N):
        if i_ads == 0:
            add_adsorbate(slab, 'H', 1, 'fcc', mol_index=0)
            pos0 = slab[-1].position
        else: #position one after the other in adjacent fcc sites
            i = int(i_ads / Nx)
            j = i_ads % Nx
            pos = i / Nx * slab.cell[0] + j / Ny * slab.cell[1] + pos0
            add_adsorbate(slab, 'H', 1, pos[0:2], mol_index=0)

    #freeze some atoms such that they cannot move during geometr optimization
    fixed = []
    for i in range(Nx*Ny*2): #fix 2 layers
        fixed.append(i)
    slab.set_constraint(FixAtoms(indices=fixed))
    slab.pbc[2]=True #3D periodic

    #retrun the geometry
    return slab

# define function that returns H2 molecule in a box
def build_H2():
    """ returns the geometry of an H2 moleucle in a box."""
    box= 10.0 # box size in Angstrom
    halfbox = box/2
    d = 0.7 # initial H-H distance in Angstrom
    H2 = Atoms('H2',
                 positions=([halfbox - d / 2, halfbox, halfbox],
                            [halfbox + d / 2, halfbox, halfbox]),
                 cell=(box, box, box))
    #return the geometry
    return H2

# define function that runs a DFT geometry optimization using the PBE functional, a single-zeta basis set and 3x3 k-points. Returns the energy
def get_opt_energy(slab, k=3):
    """ sets up a DFT geometry optimization and returns the final energy.
    Input:
    - slab: ase geometry object of the system to be optimized and to compute the energy of
    Output:
    - energy [eV]
    """
    #set up calculation in GPAW
    calc = GPAW(
                mode='lcao',
                basis = 'sz(dzp)',#my_basis,
                xc='PBE',
                kpts=(k, k, 1),
                convergence = {'eigenstates': 1.0e-6, 'density': 5.0e-4, },
            )
    print(calc)
    slab.calc = calc
    #setup and run the geometry relaxation
    relax = QuasiNewton(slab, trajectory="Pt111_{}.traj".format(sum(slab.get_tags()==0))) #Note: labels the output with the number of Hads
    #relax = BFGS(slab, trajectory='Pt111.traj')
    force_convergence_threshold = 0.07 #eV/A
    relax.run(fmax=force_convergence_threshold)
    #extract final energy
    return slab.get_potential_energy()

# !!! ----- Q1: ADJUST as needed ------ !!!
slab = build_slab_with_adsorbate(N=1, Nx=2, Ny=2, l=3)
# !!! ----- END ADJUST as needed ------ !!!
energy = get_opt_energy(slab, k=3)
print(energy)

