from ase import Atoms
from ase.constraints import FixAtoms

'''
ASE-compatible Python dictionaries with geometries for all structures
at each applied potential. The bulk-reference state for the Volmer step
into an fcc site is explicitly included, but the other reference states
are taken from other reactions. See list below for bulk reference states:

Volmer top ads: Ref. state is the final state of the Volmer step into fcc site.
Volmer two top ads: Ref. is final state of the first Volmer step in top site.
Heyrovsky: Ref. is final state of the first Volmer step in top site.
'''

# Volmer reaction into an ontop site. Initial, transition and final states
# are reported at each applied potential.
Volmer_top_ads = {

'IS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.83484549e-04,  1.62633766e+00,  8.97550012e+00],
       [ 2.81126234e+00,  1.61979091e+00,  8.96391479e+00],
       [ 5.62669645e+00,  1.61796478e+00,  8.97845314e+00],
       [ 1.40627162e+00,  4.04841429e+00,  8.96574469e+00],
       [ 4.22282827e+00,  4.05254337e+00,  8.96906677e+00],
       [ 7.01770854e+00,  4.04868471e+00,  8.98094329e+00],
       [ 7.07096507e-03,  7.85828129e-03,  1.13204387e+01],
       [ 2.81361697e+00,  2.78182159e-03,  1.13267531e+01],
       [ 5.61689845e+00,  6.66116672e-03,  1.13916157e+01],
       [ 1.41682284e+00,  2.43234352e+00,  1.13642391e+01],
       [ 4.21716517e+00,  2.43224538e+00,  1.13256793e+01],
       [ 7.02433296e+00,  2.43496315e+00,  1.14183831e+01],
       [ 5.68417025e+00, -1.32375590e-01,  1.49324841e+01],
       [ 5.53165952e+00,  4.40425726e-02,  1.39697032e+01],
       [ 6.12848737e+00,  6.76357815e-01,  1.52618115e+01],
       [ 1.30409307e+00,  2.70918618e+00,  1.49410757e+01],
       [ 1.76772194e+00,  3.55383220e+00,  1.52537387e+01],
       [ 1.55498240e+00,  2.62373046e+00,  1.39895661e+01],
       [ 7.23935200e+00,  2.67908434e+00,  1.47335315e+01],
       [ 6.69473437e+00,  3.54610484e+00,  1.49229795e+01],
       [ 8.31453908e+00,  2.78265418e+00,  1.49107260e+01],
       [ 2.63962111e+00,  2.65773774e-02,  1.57738750e+01],
       [ 3.58509009e+00,  4.54671933e-02,  1.55159400e+01],
       [ 2.28421887e+00,  9.09160755e-01,  1.55343788e+01],
       [ 7.13986035e+00,  2.52931581e+00,  1.37343636e+01],
       [ 1.41846621e+00,  8.01585597e-01,  1.22447469e+01],
       [ 4.19400580e+00,  8.18005130e-01,  1.22538583e+01],
       [ 7.04553866e+00,  7.95593901e-01,  1.22702723e+01],
       [ 3.79478397e-02,  3.27636940e+00,  1.22393030e+01],
       [ 2.82882340e+00,  3.25129275e+00,  1.22428409e+01],
       [ 5.58502658e+00,  3.23758031e+00,  1.22656001e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.22207671e-02,  1.63170059e+00,  8.98630617e+00],
       [ 2.80800878e+00,  1.61751023e+00,  8.95264760e+00],
       [ 5.63731010e+00,  1.62327887e+00,  8.98726176e+00],
       [ 1.40107306e+00,  4.04658160e+00,  8.95582056e+00],
       [ 4.22343673e+00,  4.05133128e+00,  8.95707228e+00],
       [ 7.01750961e+00,  4.04267516e+00,  8.99061211e+00],
       [ 5.42673216e-03,  7.55943362e-03,  1.12985993e+01],
       [ 2.81117032e+00,  2.00262664e-03,  1.13056869e+01],
       [ 5.61568435e+00,  5.37687120e-03,  1.13496005e+01],
       [ 1.41098670e+00,  2.43050815e+00,  1.13363110e+01],
       [ 4.21629972e+00,  2.42924538e+00,  1.12964597e+01],
       [ 7.02404380e+00,  2.43598905e+00,  1.16015878e+01],
       [ 5.60247776e+00,  5.17000534e-02,  1.49282617e+01],
       [ 5.47659491e+00,  6.72082037e-02,  1.39519370e+01],
       [ 6.10716709e+00,  8.82699457e-01,  1.51022028e+01],
       [ 1.45948012e+00,  2.65209862e+00,  1.49333261e+01],
       [ 1.91597494e+00,  3.50309714e+00,  1.52022189e+01],
       [ 1.58423824e+00,  2.60687633e+00,  1.39592300e+01],
       [ 7.10497892e+00,  2.54572537e+00,  1.48614913e+01],
       [ 6.66174630e+00,  3.42458014e+00,  1.50175384e+01],
       [ 8.08749772e+00,  2.66702780e+00,  1.50067538e+01],
       [ 2.74233325e+00,  1.13206971e-01,  1.57324818e+01],
       [ 3.69108249e+00,  1.55336264e-01,  1.54714978e+01],
       [ 2.35661205e+00,  9.79580490e-01,  1.54600364e+01],
       [ 7.05561322e+00,  2.47466586e+00,  1.32624645e+01],
       [ 1.42618188e+00,  8.05477580e-01,  1.22251622e+01],
       [ 4.18203723e+00,  8.22939805e-01,  1.22264209e+01],
       [ 7.04600053e+00,  6.83924659e-01,  1.22733034e+01],
       [ 1.35207002e-01,  3.32973253e+00,  1.22618891e+01],
       [ 2.82677665e+00,  3.25195503e+00,  1.22150195e+01],
       [ 5.47869531e+00,  3.29114443e+00,  1.22719340e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.02497335e-02,  1.62949356e+00,  8.97978362e+00],
       [ 2.80531134e+00,  1.61197652e+00,  8.94862936e+00],
       [ 5.62823668e+00,  1.62181385e+00,  8.98199735e+00],
       [ 1.40272483e+00,  4.04076707e+00,  8.95118489e+00],
       [ 4.21657491e+00,  4.04511927e+00,  8.95307823e+00],
       [ 7.01467055e+00,  4.04379979e+00,  8.98496146e+00],
       [ 9.95261882e-03,  5.38217167e-03,  1.12869744e+01],
       [ 2.80829105e+00,  3.89282173e-03,  1.13044293e+01],
       [ 5.60733907e+00,  3.07835665e-03,  1.13363941e+01],
       [ 1.41119027e+00,  2.43086729e+00,  1.13354299e+01],
       [ 4.20580982e+00,  2.42975370e+00,  1.12966274e+01],
       [ 7.01807008e+00,  2.43214933e+00,  1.15304720e+01],
       [ 5.61978685e+00,  1.45311890e-01,  1.49330702e+01],
       [ 5.57008348e+00,  1.02110409e-01,  1.39521258e+01],
       [ 6.12015747e+00,  9.87581051e-01,  1.51035976e+01],
       [ 1.48145228e+00,  2.65823917e+00,  1.49013284e+01],
       [ 1.94437844e+00,  3.51600321e+00,  1.51310217e+01],
       [ 1.51489753e+00,  2.61751617e+00,  1.39195643e+01],
       [ 7.06262795e+00,  2.53614533e+00,  1.52285564e+01],
       [ 6.65453333e+00,  3.43700671e+00,  1.52097140e+01],
       [ 8.04414528e+00,  2.66343587e+00,  1.51839689e+01],
       [ 2.76532303e+00,  1.42646794e-01,  1.56250322e+01],
       [ 3.71935949e+00,  1.93687336e-01,  1.53779734e+01],
       [ 2.37668305e+00,  1.01047644e+00,  1.53573261e+01],
       [ 7.01688140e+00,  2.43906980e+00,  1.30961557e+01],
       [ 1.41823510e+00,  8.02980393e-01,  1.22261422e+01],
       [ 4.18469012e+00,  8.23557157e-01,  1.22265164e+01],
       [ 7.04299464e+00,  6.46251335e-01,  1.22364751e+01],
       [ 1.47025771e-01,  3.34181818e+00,  1.22429745e+01],
       [ 2.82593207e+00,  3.24664311e+00,  1.22182883e+01],
       [ 5.44181943e+00,  3.31291296e+00,  1.22523841e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-2.16703677e-03,  1.62927947e+00,  8.97776471e+00],
       [ 2.81227015e+00,  1.62068264e+00,  8.96242941e+00],
       [ 5.63079246e+00,  1.61998353e+00,  8.98077605e+00],
       [ 1.40650519e+00,  4.04954473e+00,  8.96411680e+00],
       [ 4.22480713e+00,  4.05383744e+00,  8.96786249e+00],
       [ 7.01821051e+00,  4.04829393e+00,  8.98342506e+00],
       [ 7.87453005e-03,  1.01420594e-02,  1.13174792e+01],
       [ 2.81514815e+00,  3.82037554e-03,  1.13239246e+01],
       [ 5.61871457e+00,  8.88870017e-03,  1.13919728e+01],
       [ 1.41792018e+00,  2.43360274e+00,  1.13618770e+01],
       [ 4.21942241e+00,  2.43370035e+00,  1.13224276e+01],
       [ 7.02589951e+00,  2.43753270e+00,  1.14478170e+01],
       [ 5.66416814e+00, -1.25933702e-01,  1.49104262e+01],
       [ 5.52932255e+00,  4.06627573e-02,  1.39411218e+01],
       [ 6.09479432e+00,  6.91306549e-01,  1.52371776e+01],
       [ 1.33020020e+00,  2.70367609e+00,  1.49096642e+01],
       [ 1.80399236e+00,  3.54124925e+00,  1.52192033e+01],
       [ 1.56018155e+00,  2.62727437e+00,  1.39500262e+01],
       [ 7.23746781e+00,  2.66678044e+00,  1.47198282e+01],
       [ 6.69718160e+00,  3.53079574e+00,  1.49139833e+01],
       [ 8.29767225e+00,  2.77462121e+00,  1.48917277e+01],
       [ 2.67716738e+00,  4.26239336e-02,  1.57429162e+01],
       [ 3.61965731e+00,  5.33874981e-02,  1.54697399e+01],
       [ 2.31567785e+00,  9.17333471e-01,  1.54806602e+01],
       [ 7.14207766e+00,  2.52876828e+00,  1.37084192e+01],
       [ 1.42060256e+00,  8.02740913e-01,  1.22406416e+01],
       [ 4.19195317e+00,  8.20862238e-01,  1.22506468e+01],
       [ 7.04848917e+00,  7.87181946e-01,  1.22758065e+01],
       [ 4.95433786e-02,  3.28592257e+00,  1.22434050e+01],
       [ 2.83086747e+00,  3.25363445e+00,  1.22376544e+01],
       [ 5.57525832e+00,  3.24355353e+00,  1.22713990e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.24308562e-02,  1.63348646e+00,  8.98842454e+00],
       [ 2.81017529e+00,  1.62017114e+00,  8.95351444e+00],
       [ 5.64041467e+00,  1.62392292e+00,  8.99014732e+00],
       [ 1.40083144e+00,  4.04940875e+00,  8.95655985e+00],
       [ 4.22705521e+00,  4.05373887e+00,  8.95853245e+00],
       [ 7.01840360e+00,  4.04351204e+00,  8.99411421e+00],
       [ 5.30255734e-03,  1.18447339e-02,  1.13020355e+01],
       [ 2.81362178e+00,  3.46475238e-03,  1.13043439e+01],
       [ 5.61928082e+00,  8.82564053e-03,  1.13587542e+01],
       [ 1.41304979e+00,  2.43213463e+00,  1.13367250e+01],
       [ 4.22024859e+00,  2.43121176e+00,  1.12987242e+01],
       [ 7.02842916e+00,  2.44135515e+00,  1.16325321e+01],
       [ 5.61266981e+00,  4.75158854e-03,  1.49213599e+01],
       [ 5.46954905e+00,  4.90477633e-02,  1.39453248e+01],
       [ 6.11610472e+00,  8.30564719e-01,  1.51096452e+01],
       [ 1.44990185e+00,  2.65262409e+00,  1.49210415e+01],
       [ 1.89740421e+00,  3.50485452e+00,  1.52000156e+01],
       [ 1.59904777e+00,  2.60529778e+00,  1.39487148e+01],
       [ 7.12589150e+00,  2.55919629e+00,  1.47492911e+01],
       [ 6.67016030e+00,  3.43264318e+00,  1.49370559e+01],
       [ 8.10945790e+00,  2.68063858e+00,  1.49256038e+01],
       [ 2.72977347e+00,  1.10393157e-01,  1.57420384e+01],
       [ 3.67563146e+00,  1.39619350e-01,  1.54713562e+01],
       [ 2.34845467e+00,  9.74385902e-01,  1.54566115e+01],
       [ 7.06766024e+00,  2.48326947e+00,  1.33081884e+01],
       [ 1.42867695e+00,  8.06206517e-01,  1.22224751e+01],
       [ 4.18268066e+00,  8.23083091e-01,  1.22273288e+01],
       [ 7.04683026e+00,  7.02676502e-01,  1.22831896e+01],
       [ 1.32398587e-01,  3.33003333e+00,  1.22632995e+01],
       [ 2.82920967e+00,  3.25466477e+00,  1.22138204e+01],
       [ 5.49087874e+00,  3.28321494e+00,  1.22786458e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-8.56996097e-03,  1.63100799e+00,  8.98437228e+00],
       [ 2.80574887e+00,  1.61545798e+00,  8.94947530e+00],
       [ 5.63011999e+00,  1.62312933e+00,  8.98354953e+00],
       [ 1.40288486e+00,  4.04232604e+00,  8.95217516e+00],
       [ 4.21707041e+00,  4.04771917e+00,  8.95352606e+00],
       [ 7.01546735e+00,  4.04397526e+00,  8.98476139e+00],
       [ 9.32808084e-03, -1.76659557e-04,  1.12942341e+01],
       [ 2.80852921e+00,  3.82899304e-03,  1.13043817e+01],
       [ 5.60838478e+00,  9.42816917e-05,  1.13436137e+01],
       [ 1.41173656e+00,  2.42957128e+00,  1.13416073e+01],
       [ 4.20672000e+00,  2.43035954e+00,  1.12972683e+01],
       [ 7.01853797e+00,  2.42643185e+00,  1.15209309e+01],
       [ 5.62142513e+00,  1.52657481e-01,  1.49207317e+01],
       [ 5.58342033e+00,  1.08198084e-01,  1.39381427e+01],
       [ 6.12655952e+00,  9.91617076e-01,  1.50962467e+01],
       [ 1.47750573e+00,  2.65265450e+00,  1.49273930e+01],
       [ 1.95549365e+00,  3.50767256e+00,  1.51348483e+01],
       [ 1.49287230e+00,  2.60111871e+00,  1.39454210e+01],
       [ 7.06210115e+00,  2.53846101e+00,  1.53440202e+01],
       [ 6.65179606e+00,  3.43137530e+00,  1.52375377e+01],
       [ 8.03804664e+00,  2.66010063e+00,  1.52339592e+01],
       [ 2.79479820e+00,  1.37281721e-01,  1.55853826e+01],
       [ 3.74436719e+00,  1.93319219e-01,  1.53211589e+01],
       [ 2.39585179e+00,  1.00260626e+00,  1.53245868e+01],
       [ 7.01338594e+00,  2.43889588e+00,  1.30811529e+01],
       [ 1.42431352e+00,  8.06955048e-01,  1.22297801e+01],
       [ 4.18244886e+00,  8.26494760e-01,  1.22270874e+01],
       [ 7.04517638e+00,  6.28380209e-01,  1.22493709e+01],
       [ 1.60927032e-01,  3.34649275e+00,  1.22534690e+01],
       [ 2.82252093e+00,  3.25065480e+00,  1.22204100e+01],
       [ 5.42128606e+00,  3.32361574e+00,  1.22619098e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-2.88497550e-03,  1.62999519e+00,  8.97937947e+00],
       [ 2.81192892e+00,  1.62128364e+00,  8.96264093e+00],
       [ 5.63135351e+00,  1.62048541e+00,  8.98240228e+00],
       [ 1.40594656e+00,  4.04997046e+00,  8.96456954e+00],
       [ 4.22508715e+00,  4.05445963e+00,  8.96838571e+00],
       [ 7.01790920e+00,  4.04843314e+00,  8.98516912e+00],
       [ 7.14251565e-03,  9.59020295e-03,  1.13194368e+01],
       [ 2.81472993e+00,  3.72668650e-03,  1.13243399e+01],
       [ 5.61806434e+00,  8.33863730e-03,  1.13971480e+01],
       [ 1.41715094e+00,  2.43339778e+00,  1.13646313e+01],
       [ 4.21903222e+00,  2.43304840e+00,  1.13227062e+01],
       [ 7.02550165e+00,  2.43628252e+00,  1.14598203e+01],
       [ 5.66411327e+00, -1.33562399e-01,  1.49040919e+01],
       [ 5.52711904e+00,  2.13492972e-02,  1.39307481e+01],
       [ 6.10249961e+00,  6.85078083e-01,  1.52168087e+01],
       [ 1.32959327e+00,  2.70072219e+00,  1.49067048e+01],
       [ 1.79824979e+00,  3.54017184e+00,  1.52149793e+01],
       [ 1.55612365e+00,  2.62734156e+00,  1.39445670e+01],
       [ 7.22788494e+00,  2.65290309e+00,  1.47078352e+01],
       [ 6.68964403e+00,  3.51877191e+00,  1.49020631e+01],
       [ 8.28112794e+00,  2.76336906e+00,  1.48848210e+01],
       [ 2.66777833e+00,  4.66148230e-02,  1.57515121e+01],
       [ 3.60863751e+00,  5.25520505e-02,  1.54721682e+01],
       [ 2.30655998e+00,  9.18621386e-01,  1.54783347e+01],
       [ 7.13959654e+00,  2.51704255e+00,  1.36893642e+01],
       [ 1.42144422e+00,  8.02091933e-01,  1.22415322e+01],
       [ 4.19013401e+00,  8.21359143e-01,  1.22521087e+01],
       [ 7.04835116e+00,  7.82732626e-01,  1.22805856e+01],
       [ 5.21570481e-02,  3.28858895e+00,  1.22475997e+01],
       [ 2.83076435e+00,  3.25363388e+00,  1.22380368e+01],
       [ 5.57075236e+00,  3.24307871e+00,  1.22744256e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.12812595e-02,  1.63412958e+00,  8.98873343e+00],
       [ 2.81089954e+00,  1.62095373e+00,  8.95563788e+00],
       [ 5.64091616e+00,  1.62422676e+00,  8.99049766e+00],
       [ 1.40121066e+00,  4.04986712e+00,  8.95840584e+00],
       [ 4.22776722e+00,  4.05421251e+00,  8.96063913e+00],
       [ 7.01874654e+00,  4.04447998e+00,  8.99437698e+00],
       [ 5.11633925e-03,  1.27076859e-02,  1.13048332e+01],
       [ 2.81416189e+00,  3.92803729e-03,  1.13071908e+01],
       [ 5.62030637e+00,  1.00562764e-02,  1.13656846e+01],
       [ 1.41359499e+00,  2.43267568e+00,  1.13407668e+01],
       [ 4.22191705e+00,  2.43192655e+00,  1.13004564e+01],
       [ 7.02905211e+00,  2.44182325e+00,  1.16360363e+01],
       [ 5.60215426e+00, -1.69576928e-02,  1.49041461e+01],
       [ 5.46166383e+00,  4.19269093e-02,  1.39263194e+01],
       [ 6.10576673e+00,  8.03935933e-01,  1.51049729e+01],
       [ 1.43541187e+00,  2.64728095e+00,  1.49196807e+01],
       [ 1.88173617e+00,  3.49937159e+00,  1.52015370e+01],
       [ 1.59482391e+00,  2.60083481e+00,  1.39472506e+01],
       [ 7.13259123e+00,  2.56860637e+00,  1.47134980e+01],
       [ 6.66502111e+00,  3.43812372e+00,  1.49115224e+01],
       [ 8.11854157e+00,  2.68692962e+00,  1.49025737e+01],
       [ 2.71525573e+00,  1.05018386e-01,  1.57461897e+01],
       [ 3.65843819e+00,  1.28326411e-01,  1.54671686e+01],
       [ 2.33587814e+00,  9.69494287e-01,  1.54598158e+01],
       [ 7.07613467e+00,  2.49075601e+00,  1.33573772e+01],
       [ 1.42908046e+00,  8.04948107e-01,  1.22246538e+01],
       [ 4.18194110e+00,  8.23129204e-01,  1.22297386e+01],
       [ 7.04862328e+00,  7.12030082e-01,  1.22883332e+01],
       [ 1.26173761e-01,  3.32589443e+00,  1.22663447e+01],
       [ 2.82980194e+00,  3.25506395e+00,  1.22165553e+01],
       [ 5.49856545e+00,  3.27743445e+00,  1.22826906e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-7.11700893e-03,  1.63020954e+00,  8.98457277e+00],
       [ 2.80685497e+00,  1.61761792e+00,  8.95359385e+00],
       [ 5.62963534e+00,  1.62279925e+00,  8.98548174e+00],
       [ 1.40439286e+00,  4.04464068e+00,  8.95561951e+00],
       [ 4.21939342e+00,  4.05020139e+00,  8.95795084e+00],
       [ 7.01684140e+00,  4.04570381e+00,  8.98788827e+00],
       [ 7.77400619e-03,  2.38225442e-03,  1.13055912e+01],
       [ 2.80948384e+00,  8.72222208e-03,  1.13061148e+01],
       [ 5.61209775e+00,  1.21334353e-03,  1.13568966e+01],
       [ 1.40924563e+00,  2.43304361e+00,  1.13475014e+01],
       [ 4.21200833e+00,  2.43348895e+00,  1.13082096e+01],
       [ 7.01906007e+00,  2.42850662e+00,  1.15001358e+01],
       [ 5.65592682e+00,  1.70227406e-01,  1.49637936e+01],
       [ 5.61402947e+00,  1.36158203e-01,  1.39808259e+01],
       [ 6.14752956e+00,  1.01576157e+00,  1.51531136e+01],
       [ 1.47773491e+00,  2.62930392e+00,  1.49660530e+01],
       [ 1.94814523e+00,  3.48468334e+00,  1.51821814e+01],
       [ 1.48698841e+00,  2.59095096e+00,  1.39829220e+01],
       [ 7.05750299e+00,  2.55262434e+00,  1.55046530e+01],
       [ 6.64998488e+00,  3.43580163e+00,  1.53264474e+01],
       [ 8.02330740e+00,  2.65117018e+00,  1.53167061e+01],
       [ 2.79941536e+00,  1.26627715e-01,  1.56577026e+01],
       [ 3.74693032e+00,  1.88742064e-01,  1.53890768e+01],
       [ 2.39361577e+00,  9.86591661e-01,  1.53882046e+01],
       [ 7.00934890e+00,  2.43922927e+00,  1.30580664e+01],
       [ 1.42547831e+00,  8.13772752e-01,  1.22319940e+01],
       [ 4.18107538e+00,  8.30658479e-01,  1.22321707e+01],
       [ 7.04688189e+00,  6.03683763e-01,  1.22684996e+01],
       [ 1.83492540e-01,  3.35866795e+00,  1.22641645e+01],
       [ 2.82148477e+00,  3.24927731e+00,  1.22178172e+01],
       [ 5.38595091e+00,  3.34249878e+00,  1.22829890e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-3.16142412e-03,  1.63054314e+00,  8.98120942e+00],
       [ 2.81154616e+00,  1.62116417e+00,  8.96361220e+00],
       [ 5.63161168e+00,  1.62038566e+00,  8.98417939e+00],
       [ 1.40571687e+00,  4.04973824e+00,  8.96550262e+00],
       [ 4.22535904e+00,  4.05470450e+00,  8.96954222e+00],
       [ 7.01727055e+00,  4.04859302e+00,  8.98719423e+00],
       [ 6.72819502e-03,  9.56193874e-03,  1.13204121e+01],
       [ 2.81453850e+00,  3.30880232e-03,  1.13265301e+01],
       [ 5.61791562e+00,  8.27747456e-03,  1.14030844e+01],
       [ 1.41697169e+00,  2.43323487e+00,  1.13698282e+01],
       [ 4.21879740e+00,  2.43312793e+00,  1.13243232e+01],
       [ 7.02541833e+00,  2.43614997e+00,  1.14672136e+01],
       [ 5.66337942e+00, -1.33626949e-01,  1.49007752e+01],
       [ 5.53021497e+00,  1.61265794e-02,  1.39248134e+01],
       [ 6.10300918e+00,  6.85479703e-01,  1.52104638e+01],
       [ 1.32777368e+00,  2.69980966e+00,  1.49042490e+01],
       [ 1.79451740e+00,  3.53955268e+00,  1.52125829e+01],
       [ 1.54846688e+00,  2.62878108e+00,  1.39395523e+01],
       [ 7.22347773e+00,  2.65119098e+00,  1.47000311e+01],
       [ 6.68668041e+00,  3.51743403e+00,  1.48949371e+01],
       [ 8.27425849e+00,  2.76310672e+00,  1.48788145e+01],
       [ 2.66476244e+00,  4.79634882e-02,  1.57575817e+01],
       [ 3.60313068e+00,  5.05683881e-02,  1.54702986e+01],
       [ 2.30221935e+00,  9.16612267e-01,  1.54751556e+01],
       [ 7.13710203e+00,  2.51940848e+00,  1.36752151e+01],
       [ 1.42112419e+00,  8.01175536e-01,  1.22435270e+01],
       [ 4.18915862e+00,  8.21534861e-01,  1.22548832e+01],
       [ 7.04895120e+00,  7.81885863e-01,  1.22837130e+01],
       [ 5.19442081e-02,  3.28934955e+00,  1.22497195e+01],
       [ 2.83080660e+00,  3.25377904e+00,  1.22411948e+01],
       [ 5.56975728e+00,  3.24223804e+00,  1.22775573e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.06966243e-02,  1.63512201e+00,  8.98855046e+00],
       [ 2.81111574e+00,  1.62221427e+00,  8.95602893e+00],
       [ 5.64092841e+00,  1.62453531e+00,  8.99065850e+00],
       [ 1.40139091e+00,  4.05038757e+00,  8.95915819e+00],
       [ 4.22855770e+00,  4.05519025e+00,  8.96166532e+00],
       [ 7.01847493e+00,  4.04507264e+00,  8.99500355e+00],
       [ 5.29616078e-03,  1.31054954e-02,  1.13091326e+01],
       [ 2.81472644e+00,  3.75901965e-03,  1.13090823e+01],
       [ 5.62009458e+00,  1.11654470e-02,  1.13757076e+01],
       [ 1.41449807e+00,  2.43326282e+00,  1.13454765e+01],
       [ 4.22273594e+00,  2.43288988e+00,  1.13035926e+01],
       [ 7.02932352e+00,  2.44232744e+00,  1.16214856e+01],
       [ 5.59968737e+00, -4.22122593e-02,  1.48934283e+01],
       [ 5.46163086e+00,  3.02685499e-02,  1.39139958e+01],
       [ 6.10192438e+00,  7.74848930e-01,  1.51079203e+01],
       [ 1.41764020e+00,  2.64439642e+00,  1.49172982e+01],
       [ 1.86268623e+00,  3.49529968e+00,  1.52042834e+01],
       [ 1.58748867e+00,  2.59633372e+00,  1.39451763e+01],
       [ 7.14478763e+00,  2.58059256e+00,  1.46830169e+01],
       [ 6.66718945e+00,  3.44776164e+00,  1.48840476e+01],
       [ 8.13426481e+00,  2.69836267e+00,  1.48788547e+01],
       [ 2.71034153e+00,  9.68738611e-02,  1.57549490e+01],
       [ 3.64924366e+00,  1.15516473e-01,  1.54621777e+01],
       [ 2.32847843e+00,  9.58151393e-01,  1.54630901e+01],
       [ 7.08346610e+00,  2.49346725e+00,  1.33845657e+01],
       [ 1.42928885e+00,  8.04003066e-01,  1.22271154e+01],
       [ 4.18136439e+00,  8.23330236e-01,  1.22328807e+01],
       [ 7.04956761e+00,  7.19334667e-01,  1.22921566e+01],
       [ 1.22552532e-01,  3.32471011e+00,  1.22668744e+01],
       [ 2.82994365e+00,  3.25554472e+00,  1.22191633e+01],
       [ 5.50310704e+00,  3.27267993e+00,  1.22847011e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-2.34757640e-03,  1.63275559e+00,  8.97864759e+00],
       [ 2.81011393e+00,  1.62605527e+00,  8.96086972e+00],
       [ 5.62868947e+00,  1.62419658e+00,  8.98523791e+00],
       [ 1.40693445e+00,  4.05233987e+00,  8.96337913e+00],
       [ 4.22434418e+00,  4.05632334e+00,  8.96691187e+00],
       [ 7.01916141e+00,  4.05305559e+00,  8.98537389e+00],
       [ 5.68246148e-03,  8.30592396e-03,  1.13214421e+01],
       [ 2.81366088e+00,  1.61139804e-02,  1.13050398e+01],
       [ 5.62040339e+00,  3.48838371e-03,  1.13863937e+01],
       [ 1.40778983e+00,  2.43894009e+00,  1.13585124e+01],
       [ 4.22121200e+00,  2.43950992e+00,  1.13260447e+01],
       [ 7.02104276e+00,  2.43137006e+00,  1.14481461e+01],
       [ 5.70205856e+00,  1.73902255e-01,  1.50505405e+01],
       [ 5.67780776e+00,  1.47202411e-01,  1.40672642e+01],
       [ 6.18693391e+00,  1.02106924e+00,  1.52573340e+01],
       [ 1.47667994e+00,  2.61276463e+00,  1.49917369e+01],
       [ 1.95349014e+00,  3.46525979e+00,  1.52027068e+01],
       [ 1.46905308e+00,  2.57827216e+00,  1.40078999e+01],
       [ 7.07283901e+00,  2.55113553e+00,  1.56688778e+01],
       [ 6.66257043e+00,  3.42116198e+00,  1.54410232e+01],
       [ 8.02475359e+00,  2.62962434e+00,  1.54149183e+01],
       [ 2.80136157e+00,  1.16916794e-01,  1.56642974e+01],
       [ 3.75304825e+00,  1.84026426e-01,  1.54157781e+01],
       [ 2.39841543e+00,  9.78986910e-01,  1.53966130e+01],
       [ 7.00230320e+00,  2.44010826e+00,  1.30065547e+01],
       [ 1.42946317e+00,  8.24754619e-01,  1.22395622e+01],
       [ 4.18692303e+00,  8.45089665e-01,  1.22460313e+01],
       [ 7.05396354e+00,  5.25359515e-01,  1.23120074e+01],
       [ 2.18943122e-01,  3.37987119e+00,  1.22840312e+01],
       [ 2.81679852e+00,  3.25667500e+00,  1.22238615e+01],
       [ 5.31030176e+00,  3.38770147e+00,  1.23305482e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5]))
}

# Second Volmer reaction into an ontop site. Initial, transition and final
# states are reported at each applied potential.
Volmer_2top_ads = {

'IS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-6.48032825e-03,  1.65470451e+00,  8.97541167e+00],
       [ 2.78806973e+00,  1.61918406e+00,  8.96034213e+00],
       [ 5.62112912e+00,  1.61343682e+00,  8.96446155e+00],
       [ 1.39917338e+00,  4.05867129e+00,  8.97422059e+00],
       [ 4.20740495e+00,  4.05779230e+00,  8.96316993e+00],
       [ 7.00060773e+00,  4.06273843e+00,  8.98045480e+00],
       [-1.29827678e-02,  5.57769604e-02,  1.12672606e+01],
       [ 2.81601108e+00,  8.40756811e-03,  1.13642225e+01],
       [ 5.57564403e+00,  8.50063308e-03,  1.14560910e+01],
       [ 1.42158416e+00,  2.46750078e+00,  1.14229078e+01],
       [ 4.20382383e+00,  2.40854924e+00,  1.12542447e+01],
       [ 6.98477070e+00,  2.46591008e+00,  1.13853838e+01],
       [ 5.36801759e+00, -1.40667754e-01,  1.50782898e+01],
       [ 5.38925887e+00,  7.54660454e-03,  1.41027346e+01],
       [ 5.78960908e+00,  6.61895687e-01,  1.54504894e+01],
       [ 1.39768618e+00,  2.59619959e+00,  1.49064598e+01],
       [ 1.95064830e+00,  3.49711921e+00,  1.52843761e+01],
       [ 1.52893871e+00,  2.54182292e+00,  1.39145379e+01],
       [ 7.23474289e+00,  2.52459832e+00,  1.49614815e+01],
       [ 6.68630649e+00,  3.31861602e+00,  1.51897032e+01],
       [ 8.78763458e+00,  2.66477936e+00,  1.50388877e+01],
       [ 2.65966647e+00, -3.32338335e-01,  1.57849686e+01],
       [ 3.62346770e+00, -2.74586492e-01,  1.55270511e+01],
       [ 2.26063214e+00,  5.38043814e-01,  1.55809087e+01],
       [ 7.06775657e+00,  2.40222619e+00,  1.39949914e+01],
       [ 1.37325744e+00,  8.59705193e-01,  1.22490592e+01],
       [ 4.14750307e+00,  1.45371183e-01,  1.24712563e+01],
       [ 7.03652680e+00,  8.62556403e-01,  1.22539429e+01],
       [-1.32569933e-02,  3.27367998e+00,  1.21985505e+01],
       [ 2.34505644e+00,  3.54953685e+00,  1.24144415e+01],
       [ 6.07373381e+00,  3.49685161e+00,  1.24319969e+01],
       [ 4.19832077e+00,  2.34329825e+00,  1.28145365e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.11146872e-02,  1.65306797e+00,  8.98563778e+00],
       [ 2.79658108e+00,  1.60704405e+00,  8.94664392e+00],
       [ 5.63505992e+00,  1.61885930e+00,  8.98035311e+00],
       [ 1.40378497e+00,  4.04654256e+00,  8.95624510e+00],
       [ 4.21294053e+00,  4.04721622e+00,  8.94878494e+00],
       [ 7.00389874e+00,  4.04889058e+00,  8.99046384e+00],
       [ 1.40687440e-03,  3.02644484e-02,  1.12367626e+01],
       [ 2.81770477e+00, -9.21041320e-03,  1.13521860e+01],
       [ 5.57605232e+00, -1.01577125e-02,  1.13756722e+01],
       [ 1.43335519e+00,  2.44775447e+00,  1.13654226e+01],
       [ 4.20660481e+00,  2.39080198e+00,  1.12498607e+01],
       [ 7.00145615e+00,  2.42669696e+00,  1.15947788e+01],
       [ 5.56688313e+00, -5.55436630e-04,  1.50898799e+01],
       [ 5.45363955e+00,  6.82793131e-03,  1.41150556e+01],
       [ 6.08109115e+00,  8.33039104e-01,  1.52566158e+01],
       [ 1.47372780e+00,  2.40439796e+00,  1.50881087e+01],
       [ 1.93066289e+00,  3.25825226e+00,  1.53488950e+01],
       [ 1.55610728e+00,  2.37721317e+00,  1.41109049e+01],
       [ 7.04685950e+00,  2.38532165e+00,  1.51295996e+01],
       [ 6.64507579e+00,  3.28253033e+00,  1.52411160e+01],
       [ 8.02894250e+00,  2.48537100e+00,  1.52272048e+01],
       [ 2.72737324e+00, -1.23691779e-01,  1.58373518e+01],
       [ 3.68298029e+00, -4.06076722e-02,  1.56094662e+01],
       [ 2.32618418e+00,  7.43174860e-01,  1.55861128e+01],
       [ 7.00866260e+00,  2.35640046e+00,  1.31746763e+01],
       [ 1.40861018e+00,  8.59898287e-01,  1.22291717e+01],
       [ 4.16451567e+00,  1.55531639e-01,  1.24304469e+01],
       [ 7.07321076e+00,  6.80385627e-01,  1.22298116e+01],
       [ 9.52105895e-02,  3.33029941e+00,  1.22076919e+01],
       [ 2.34462585e+00,  3.51607367e+00,  1.23932475e+01],
       [ 5.89491849e+00,  3.55789754e+00,  1.24150949e+01],
       [ 4.15260792e+00,  2.31557971e+00,  1.28085461e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.96385828e-03,  1.65271522e+00,  8.98505635e+00],
       [ 2.80366607e+00,  1.61696390e+00,  8.95939264e+00],
       [ 5.63214882e+00,  1.63659447e+00,  8.97856452e+00],
       [ 1.40936375e+00,  4.05462313e+00,  8.96619001e+00],
       [ 4.21630941e+00,  4.05682266e+00,  8.96260321e+00],
       [ 7.01359605e+00,  4.06232909e+00,  8.98758331e+00],
       [-5.87611840e-03,  3.06855287e-02,  1.12737201e+01],
       [ 2.80967433e+00, -3.67234888e-03,  1.13563186e+01],
       [ 5.58485328e+00,  1.86240374e-02,  1.13651805e+01],
       [ 1.42588115e+00,  2.45170253e+00,  1.13800271e+01],
       [ 4.18374054e+00,  2.39972389e+00,  1.13079152e+01],
       [ 7.02402953e+00,  2.42453693e+00,  1.15028602e+01],
       [ 5.57372900e+00, -3.08872427e-03,  1.50515026e+01],
       [ 5.54685420e+00, -2.76338738e-03,  1.40705885e+01],
       [ 6.06230146e+00,  8.39013479e-01,  1.52704965e+01],
       [ 1.40351048e+00,  2.43367184e+00,  1.50980807e+01],
       [ 1.88171433e+00,  3.27917772e+00,  1.53361515e+01],
       [ 1.42383023e+00,  2.41295075e+00,  1.41176490e+01],
       [ 6.95074697e+00,  2.35214486e+00,  1.56101323e+01],
       [ 6.55379637e+00,  3.24030808e+00,  1.54390149e+01],
       [ 7.92115623e+00,  2.44753086e+00,  1.54534207e+01],
       [ 2.74809304e+00, -7.90873606e-02,  1.58110387e+01],
       [ 3.69451866e+00, -1.13499271e-02,  1.55389615e+01],
       [ 2.34022084e+00,  7.81304891e-01,  1.55505427e+01],
       [ 7.01392923e+00,  2.35647493e+00,  1.30615161e+01],
       [ 1.40342351e+00,  8.64420602e-01,  1.22497850e+01],
       [ 4.17955695e+00,  1.89244051e-01,  1.24206773e+01],
       [ 7.04756702e+00,  5.30821381e-01,  1.22833655e+01],
       [ 8.49249207e-02,  3.33928000e+00,  1.22191308e+01],
       [ 2.33112357e+00,  3.53743195e+00,  1.24074842e+01],
       [ 5.57413092e+00,  3.29683123e+00,  1.21730993e+01],
       [ 4.03232486e+00,  2.27485852e+00,  1.28572240e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-5.05539431e-03,  1.65518658e+00,  8.97613384e+00],
       [ 2.78819010e+00,  1.62014483e+00,  8.95960243e+00],
       [ 5.61901056e+00,  1.61306839e+00,  8.96530068e+00],
       [ 1.39935668e+00,  4.05910712e+00,  8.97314458e+00],
       [ 4.20870600e+00,  4.05792262e+00,  8.96172796e+00],
       [ 6.99899985e+00,  4.06301394e+00,  8.97978795e+00],
       [-1.26331973e-02,  5.60424615e-02,  1.12689383e+01],
       [ 2.81564728e+00,  9.22952974e-03,  1.13651686e+01],
       [ 5.57483348e+00,  1.00200613e-02,  1.14591207e+01],
       [ 1.42171190e+00,  2.46700339e+00,  1.14302048e+01],
       [ 4.20339232e+00,  2.40812750e+00,  1.12538520e+01],
       [ 6.98455296e+00,  2.46626086e+00,  1.13898892e+01],
       [ 5.36973501e+00, -1.39485512e-01,  1.50777719e+01],
       [ 5.38935053e+00,  9.16806353e-03,  1.41013611e+01],
       [ 5.79034205e+00,  6.64506234e-01,  1.54481566e+01],
       [ 1.39856050e+00,  2.59575448e+00,  1.49034740e+01],
       [ 1.95097935e+00,  3.49065017e+00,  1.52826192e+01],
       [ 1.52865538e+00,  2.54144584e+00,  1.39088432e+01],
       [ 7.23217255e+00,  2.52080951e+00,  1.49627116e+01],
       [ 6.68995942e+00,  3.31982907e+00,  1.51894334e+01],
       [ 8.78851567e+00,  2.66248246e+00,  1.50402538e+01],
       [ 2.66109657e+00, -3.31233683e-01,  1.57901806e+01],
       [ 3.62211495e+00, -2.71637208e-01,  1.55238217e+01],
       [ 2.26161285e+00,  5.37585593e-01,  1.55804213e+01],
       [ 7.07110258e+00,  2.40323648e+00,  1.39934221e+01],
       [ 1.37214693e+00,  8.59880487e-01,  1.22488227e+01],
       [ 4.14748320e+00,  1.44775083e-01,  1.24743657e+01],
       [ 7.03682440e+00,  8.61645915e-01,  1.22563981e+01],
       [-1.27201124e-02,  3.27351278e+00,  1.22003660e+01],
       [ 2.34399657e+00,  3.55097434e+00,  1.24161078e+01],
       [ 6.07411600e+00,  3.49680667e+00,  1.24345576e+01],
       [ 4.19747996e+00,  2.34284162e+00,  1.28146172e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.58856305e-02,  1.65662885e+00,  8.98856044e+00],
       [ 2.79658344e+00,  1.61541267e+00,  8.94872195e+00],
       [ 5.63960526e+00,  1.61908403e+00,  8.98785470e+00],
       [ 1.39635155e+00,  4.05563460e+00,  8.96175183e+00],
       [ 4.21774279e+00,  4.05346924e+00,  8.95518780e+00],
       [ 6.99972725e+00,  4.05069998e+00,  8.99708017e+00],
       [-1.16590682e-02,  4.15613656e-02,  1.12500339e+01],
       [ 2.82053348e+00, -5.44635731e-03,  1.13464367e+01],
       [ 5.57963348e+00,  3.72128368e-03,  1.14172906e+01],
       [ 1.42792391e+00,  2.45142599e+00,  1.13600313e+01],
       [ 4.21846493e+00,  2.39905164e+00,  1.12486665e+01],
       [ 6.99947407e+00,  2.44512042e+00,  1.16689990e+01],
       [ 5.57249114e+00, -1.17072749e-01,  1.50947684e+01],
       [ 5.40969400e+00, -3.28113553e-03,  1.41309336e+01],
       [ 6.07275762e+00,  6.98385423e-01,  1.53232773e+01],
       [ 1.42840300e+00,  2.41025936e+00,  1.50243126e+01],
       [ 1.87773119e+00,  3.25932402e+00,  1.53119268e+01],
       [ 1.60243224e+00,  2.36514514e+00,  1.40570982e+01],
       [ 7.12128549e+00,  2.42729392e+00,  1.47919558e+01],
       [ 6.68476370e+00,  3.29309787e+00,  1.50356692e+01],
       [ 8.10584778e+00,  2.51782233e+00,  1.49853007e+01],
       [ 2.68786723e+00, -1.39858632e-01,  1.58281775e+01],
       [ 3.64602632e+00, -7.84170038e-02,  1.56136996e+01],
       [ 2.30119497e+00,  7.26965954e-01,  1.55589893e+01],
       [ 7.04964755e+00,  2.40458996e+00,  1.33630261e+01],
       [ 1.41779910e+00,  8.61939528e-01,  1.22212891e+01],
       [ 4.14867220e+00,  1.53683796e-01,  1.24476789e+01],
       [ 7.08736351e+00,  7.31070361e-01,  1.22795052e+01],
       [ 9.10382514e-02,  3.33400817e+00,  1.22282090e+01],
       [ 2.35054705e+00,  3.51690084e+00,  1.23800760e+01],
       [ 5.94071131e+00,  3.58963733e+00,  1.24806691e+01],
       [ 4.16372324e+00,  2.32880046e+00,  1.28084666e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.00162027e-03,  1.65275229e+00,  8.98536797e+00],
       [ 2.80373243e+00,  1.61687322e+00,  8.95969026e+00],
       [ 5.63216013e+00,  1.63660271e+00,  8.97864212e+00],
       [ 1.40943814e+00,  4.05459334e+00,  8.96637000e+00],
       [ 4.21624672e+00,  4.05684856e+00,  8.96265763e+00],
       [ 7.01362564e+00,  4.06238322e+00,  8.98776648e+00],
       [-5.95886248e-03,  3.06527156e-02,  1.12739697e+01],
       [ 2.80974241e+00, -3.61464276e-03,  1.13563717e+01],
       [ 5.58488397e+00,  1.86337170e-02,  1.13654967e+01],
       [ 1.42594473e+00,  2.45174811e+00,  1.13803078e+01],
       [ 4.18357887e+00,  2.39964284e+00,  1.13080217e+01],
       [ 7.02430272e+00,  2.42459835e+00,  1.15030112e+01],
       [ 5.57371573e+00, -3.21919218e-03,  1.50518197e+01],
       [ 5.54688655e+00, -2.64044294e-03,  1.40704136e+01],
       [ 6.06219138e+00,  8.38929465e-01,  1.52702094e+01],
       [ 1.40359998e+00,  2.43377037e+00,  1.50983028e+01],
       [ 1.88168738e+00,  3.27912475e+00,  1.53361646e+01],
       [ 1.42374856e+00,  2.41285427e+00,  1.41171172e+01],
       [ 6.95072059e+00,  2.35196757e+00,  1.56103465e+01],
       [ 6.55384048e+00,  3.24032457e+00,  1.54388810e+01],
       [ 7.92130242e+00,  2.44748512e+00,  1.54532603e+01],
       [ 2.74818913e+00, -7.90615118e-02,  1.58112767e+01],
       [ 3.69463138e+00, -1.13400075e-02,  1.55388044e+01],
       [ 2.34032652e+00,  7.81402045e-01,  1.55502830e+01],
       [ 7.01384729e+00,  2.35650901e+00,  1.30612717e+01],
       [ 1.40348858e+00,  8.64361406e-01,  1.22499417e+01],
       [ 4.17962981e+00,  1.89247510e-01,  1.24206863e+01],
       [ 7.04753691e+00,  5.30853936e-01,  1.22833810e+01],
       [ 8.48676981e-02,  3.33923340e+00,  1.22191358e+01],
       [ 2.33118388e+00,  3.53737493e+00,  1.24075560e+01],
       [ 5.57411601e+00,  3.29683527e+00,  1.21730800e+01],
       [ 4.03232487e+00,  2.27481580e+00,  1.28573828e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-5.00167383e-03,  1.65845092e+00,  8.98209311e+00],
       [ 2.78618840e+00,  1.62218756e+00,  8.96248267e+00],
       [ 5.62085243e+00,  1.61450510e+00,  8.96814685e+00],
       [ 1.39877333e+00,  4.05968262e+00,  8.97702647e+00],
       [ 4.20872427e+00,  4.06007327e+00,  8.96380130e+00],
       [ 6.99889976e+00,  4.06390199e+00,  8.98398301e+00],
       [-1.36760896e-02,  5.70735301e-02,  1.12708237e+01],
       [ 2.81746610e+00,  8.83609955e-03,  1.13672196e+01],
       [ 5.57608460e+00,  9.99783458e-03,  1.14729108e+01],
       [ 1.42422194e+00,  2.46887621e+00,  1.14531737e+01],
       [ 4.20380693e+00,  2.40913004e+00,  1.12518133e+01],
       [ 6.98531582e+00,  2.46805696e+00,  1.14020610e+01],
       [ 5.37462352e+00, -1.35717763e-01,  1.50741838e+01],
       [ 5.39053789e+00,  1.26206693e-03,  1.40941932e+01],
       [ 5.79358155e+00,  6.75741698e-01,  1.54305101e+01],
       [ 1.39975067e+00,  2.58804472e+00,  1.48926664e+01],
       [ 1.94666758e+00,  3.46825316e+00,  1.52713112e+01],
       [ 1.52354083e+00,  2.54688001e+00,  1.38907920e+01],
       [ 7.23839687e+00,  2.50920823e+00,  1.49571923e+01],
       [ 6.70215786e+00,  3.31144496e+00,  1.51854394e+01],
       [ 8.78780664e+00,  2.65275641e+00,  1.50332430e+01],
       [ 2.67262311e+00, -3.35097625e-01,  1.57952098e+01],
       [ 3.63165332e+00, -2.67310388e-01,  1.55236639e+01],
       [ 2.26607988e+00,  5.29721645e-01,  1.55830830e+01],
       [ 7.08237532e+00,  2.40156348e+00,  1.39837560e+01],
       [ 1.36788464e+00,  8.54112605e-01,  1.22570703e+01],
       [ 4.14229936e+00,  1.41348593e-01,  1.24821436e+01],
       [ 7.03919309e+00,  8.60158966e-01,  1.22665779e+01],
       [-1.45170710e-02,  3.27965292e+00,  1.22121629e+01],
       [ 2.34549414e+00,  3.56252058e+00,  1.24306180e+01],
       [ 6.07414803e+00,  3.49757922e+00,  1.24492134e+01],
       [ 4.19372055e+00,  2.34206557e+00,  1.28131639e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.48934026e-02,  1.65695416e+00,  8.98803449e+00],
       [ 2.79701459e+00,  1.61505554e+00,  8.94853822e+00],
       [ 5.64000304e+00,  1.61800352e+00,  8.98588336e+00],
       [ 1.39738775e+00,  4.05560017e+00,  8.96197191e+00],
       [ 4.21879996e+00,  4.05405980e+00,  8.95469010e+00],
       [ 7.00014972e+00,  4.05047897e+00,  8.99636420e+00],
       [-1.17137909e-02,  4.40541508e-02,  1.12510441e+01],
       [ 2.82090928e+00, -4.28916250e-03,  1.13476486e+01],
       [ 5.57921027e+00,  7.25384124e-03,  1.14235119e+01],
       [ 1.42901865e+00,  2.45172534e+00,  1.13639958e+01],
       [ 4.21941878e+00,  2.39950119e+00,  1.12470663e+01],
       [ 7.00083910e+00,  2.44728884e+00,  1.16521882e+01],
       [ 5.57146182e+00, -1.43030277e-01,  1.50852375e+01],
       [ 5.41531149e+00, -2.04425563e-02,  1.41191462e+01],
       [ 6.07847143e+00,  6.65280631e-01,  1.53217346e+01],
       [ 1.42039089e+00,  2.41705050e+00,  1.50208151e+01],
       [ 1.87023093e+00,  3.26506224e+00,  1.53116833e+01],
       [ 1.60588262e+00,  2.36808671e+00,  1.40546480e+01],
       [ 7.13720786e+00,  2.43311959e+00,  1.47450440e+01],
       [ 6.68600041e+00,  3.29243130e+00,  1.49952811e+01],
       [ 8.12374990e+00,  2.52341558e+00,  1.49498397e+01],
       [ 2.68810725e+00, -1.38332539e-01,  1.58336464e+01],
       [ 3.64445848e+00, -8.30057592e-02,  1.56091534e+01],
       [ 2.30319254e+00,  7.28501088e-01,  1.55634057e+01],
       [ 7.06255900e+00,  2.41145115e+00,  1.33851926e+01],
       [ 1.41583900e+00,  8.61417524e-01,  1.22216772e+01],
       [ 4.14775587e+00,  1.52198008e-01,  1.24505585e+01],
       [ 7.08617310e+00,  7.37711641e-01,  1.22806798e+01],
       [ 8.83017460e-02,  3.33308853e+00,  1.22263656e+01],
       [ 2.34887064e+00,  3.51776361e+00,  1.23834576e+01],
       [ 5.94468837e+00,  3.57983511e+00,  1.24773844e+01],
       [ 4.16427733e+00,  2.32862867e+00,  1.28068560e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.77320175e-03,  1.65246029e+00,  8.98299688e+00],
       [ 2.80324918e+00,  1.61746525e+00,  8.95736691e+00],
       [ 5.63194865e+00,  1.63655641e+00,  8.97812467e+00],
       [ 1.40892251e+00,  4.05481784e+00,  8.96477470e+00],
       [ 4.21662133e+00,  4.05663092e+00,  8.96202605e+00],
       [ 7.01333780e+00,  4.06190937e+00,  8.98631790e+00],
       [-5.63244180e-03,  3.06923912e-02,  1.12728165e+01],
       [ 2.80937935e+00, -3.97545924e-03,  1.13565913e+01],
       [ 5.58489418e+00,  1.85198394e-02,  1.13659234e+01],
       [ 1.42529873e+00,  2.45139379e+00,  1.13808427e+01],
       [ 4.18458032e+00,  2.40009138e+00,  1.13062602e+01],
       [ 7.02245916e+00,  2.42436312e+00,  1.15018419e+01],
       [ 5.57363597e+00, -2.31145993e-03,  1.50513181e+01],
       [ 5.54646229e+00, -3.74853464e-03,  1.40682359e+01],
       [ 6.06279560e+00,  8.39239573e-01,  1.52707121e+01],
       [ 1.40326657e+00,  2.43322385e+00,  1.50989299e+01],
       [ 1.88145331e+00,  3.27860730e+00,  1.53348277e+01],
       [ 1.42414576e+00,  2.41327998e+00,  1.41167469e+01],
       [ 6.95099395e+00,  2.35369312e+00,  1.56153921e+01],
       [ 6.55328507e+00,  3.24040391e+00,  1.54374041e+01],
       [ 7.92040415e+00,  2.44793568e+00,  1.54520769e+01],
       [ 2.74796952e+00, -7.86455215e-02,  1.58130480e+01],
       [ 3.69394706e+00, -1.14194817e-02,  1.55381277e+01],
       [ 2.33919405e+00,  7.81066837e-01,  1.55505638e+01],
       [ 7.01454394e+00,  2.35614915e+00,  1.30611358e+01],
       [ 1.40276674e+00,  8.64902464e-01,  1.22483204e+01],
       [ 4.17905135e+00,  1.88905925e-01,  1.24203352e+01],
       [ 7.04790330e+00,  5.29731856e-01,  1.22825175e+01],
       [ 8.58014750e-02,  3.34017872e+00,  1.22180976e+01],
       [ 2.33041005e+00,  3.53792973e+00,  1.24066453e+01],
       [ 5.57419184e+00,  3.29741077e+00,  1.21720899e+01],
       [ 4.03209709e+00,  2.27507412e+00,  1.28564713e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-5.05965733e-03,  1.65846591e+00,  8.98210872e+00],
       [ 2.78615485e+00,  1.62212025e+00,  8.96250684e+00],
       [ 5.62097026e+00,  1.61450714e+00,  8.96813954e+00],
       [ 1.39874546e+00,  4.05975364e+00,  8.97711638e+00],
       [ 4.20869890e+00,  4.06012456e+00,  8.96377367e+00],
       [ 6.99896731e+00,  4.06391690e+00,  8.98397633e+00],
       [-1.35004218e-02,  5.72023744e-02,  1.12709106e+01],
       [ 2.81729676e+00,  9.05841830e-03,  1.13672703e+01],
       [ 5.57610128e+00,  1.01524475e-02,  1.14731430e+01],
       [ 1.42415561e+00,  2.46867279e+00,  1.14534067e+01],
       [ 4.20388396e+00,  2.40901893e+00,  1.12518108e+01],
       [ 6.98532658e+00,  2.46788117e+00,  1.14022800e+01],
       [ 5.37460751e+00, -1.35754812e-01,  1.50748728e+01],
       [ 5.39052236e+00,  1.09431066e-03,  1.40934549e+01],
       [ 5.79361339e+00,  6.75688961e-01,  1.54303063e+01],
       [ 1.39986558e+00,  2.58831881e+00,  1.48937294e+01],
       [ 1.94632755e+00,  3.46794918e+00,  1.52712000e+01],
       [ 1.52363097e+00,  2.54671229e+00,  1.38899080e+01],
       [ 7.23837948e+00,  2.50916688e+00,  1.49578518e+01],
       [ 6.70230244e+00,  3.31133384e+00,  1.51853072e+01],
       [ 8.78803107e+00,  2.65281953e+00,  1.50331421e+01],
       [ 2.67271442e+00, -3.35140209e-01,  1.57955600e+01],
       [ 3.63171136e+00, -2.67328140e-01,  1.55235295e+01],
       [ 2.26624271e+00,  5.29648788e-01,  1.55829684e+01],
       [ 7.08236488e+00,  2.40149055e+00,  1.39829570e+01],
       [ 1.36787448e+00,  8.54152358e-01,  1.22569567e+01],
       [ 4.14233573e+00,  1.41321799e-01,  1.24821149e+01],
       [ 7.03918563e+00,  8.60212987e-01,  1.22664974e+01],
       [-1.44804347e-02,  3.27970794e+00,  1.22121229e+01],
       [ 2.34545176e+00,  3.56256819e+00,  1.24306808e+01],
       [ 6.07417803e+00,  3.49757668e+00,  1.24491930e+01],
       [ 4.19370307e+00,  2.34205675e+00,  1.28131660e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.40679302e-02,  1.65795413e+00,  8.98813915e+00],
       [ 2.79755034e+00,  1.61537248e+00,  8.94933557e+00],
       [ 5.64002385e+00,  1.61842022e+00,  8.98521314e+00],
       [ 1.39833907e+00,  4.05620516e+00,  8.96301483e+00],
       [ 4.21896686e+00,  4.05475835e+00,  8.95535597e+00],
       [ 7.00054408e+00,  4.05149342e+00,  8.99643880e+00],
       [-1.11948607e-02,  4.51066802e-02,  1.12520948e+01],
       [ 2.82100743e+00, -4.19114600e-03,  1.13504852e+01],
       [ 5.57955399e+00,  7.59714310e-03,  1.14292891e+01],
       [ 1.43076043e+00,  2.45365682e+00,  1.13679899e+01],
       [ 4.21980121e+00,  2.40068594e+00,  1.12472579e+01],
       [ 7.00220630e+00,  2.45021453e+00,  1.16437042e+01],
       [ 5.57088760e+00, -1.61590153e-01,  1.50807343e+01],
       [ 5.41681057e+00, -3.23618982e-02,  1.41143998e+01],
       [ 6.07655101e+00,  6.43588217e-01,  1.53257099e+01],
       [ 1.40983133e+00,  2.42233315e+00,  1.50105158e+01],
       [ 1.86161058e+00,  3.26747876e+00,  1.53069972e+01],
       [ 1.60787885e+00,  2.37207560e+00,  1.40459678e+01],
       [ 7.14520948e+00,  2.43771468e+00,  1.47251739e+01],
       [ 6.68543118e+00,  3.29445589e+00,  1.49774356e+01],
       [ 8.13603194e+00,  2.52799823e+00,  1.49288692e+01],
       [ 2.68601148e+00, -1.38953186e-01,  1.58360244e+01],
       [ 3.64096590e+00, -8.91971793e-02,  1.56045035e+01],
       [ 2.30253464e+00,  7.27208902e-01,  1.55625254e+01],
       [ 7.07111308e+00,  2.41407112e+00,  1.34041432e+01],
       [ 1.41310463e+00,  8.61641863e-01,  1.22232799e+01],
       [ 4.14662234e+00,  1.50866281e-01,  1.24547793e+01],
       [ 7.08338043e+00,  7.43593212e-01,  1.22799012e+01],
       [ 8.51312002e-02,  3.33327487e+00,  1.22244120e+01],
       [ 2.34772454e+00,  3.52021074e+00,  1.23865250e+01],
       [ 5.94613067e+00,  3.57019605e+00,  1.24731279e+01],
       [ 4.16419086e+00,  2.32813865e+00,  1.28074435e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.43223990e-03,  1.65159204e+00,  8.98090902e+00],
       [ 2.80197120e+00,  1.61714858e+00,  8.95639886e+00],
       [ 5.63112936e+00,  1.63522709e+00,  8.97737870e+00],
       [ 1.40865556e+00,  4.05381616e+00,  8.96394186e+00],
       [ 4.21608125e+00,  4.05688247e+00,  8.96119420e+00],
       [ 7.01157333e+00,  4.06037415e+00,  8.98506317e+00],
       [-6.79568034e-03,  3.12142579e-02,  1.12667106e+01],
       [ 2.80883205e+00, -3.82985847e-03,  1.13536140e+01],
       [ 5.58383955e+00,  1.94597627e-02,  1.13690690e+01],
       [ 1.42456105e+00,  2.45155706e+00,  1.13820222e+01],
       [ 4.18303771e+00,  2.39979638e+00,  1.13007102e+01],
       [ 7.02198715e+00,  2.42464997e+00,  1.14955633e+01],
       [ 5.57237749e+00, -1.26651687e-03,  1.50456297e+01],
       [ 5.54607868e+00, -3.47312425e-03,  1.40621587e+01],
       [ 6.06293240e+00,  8.39012148e-01,  1.52650618e+01],
       [ 1.40276727e+00,  2.43204596e+00,  1.50939362e+01],
       [ 1.88021856e+00,  3.27698537e+00,  1.53316726e+01],
       [ 1.42426648e+00,  2.41200574e+00,  1.41114317e+01],
       [ 6.95424478e+00,  2.36154097e+00,  1.56440275e+01],
       [ 6.55131267e+00,  3.23824025e+00,  1.54306157e+01],
       [ 7.91798114e+00,  2.44928392e+00,  1.54458796e+01],
       [ 2.74602524e+00, -7.57690762e-02,  1.58213076e+01],
       [ 3.68860024e+00, -1.15278355e-02,  1.55341274e+01],
       [ 2.33444707e+00,  7.79731637e-01,  1.55494573e+01],
       [ 7.01694830e+00,  2.35410060e+00,  1.30547783e+01],
       [ 1.39934657e+00,  8.65197019e-01,  1.22443433e+01],
       [ 4.17617818e+00,  1.86871978e-01,  1.24205173e+01],
       [ 7.05031507e+00,  5.22763882e-01,  1.22803892e+01],
       [ 8.81982728e-02,  3.34419671e+00,  1.22131733e+01],
       [ 2.32735528e+00,  3.54058612e+00,  1.24052774e+01],
       [ 5.57406635e+00,  3.29874614e+00,  1.21681203e+01],
       [ 4.03087217e+00,  2.27596900e+00,  1.28507680e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5]))
}

# Volmer reaction into an fcc site. Initial, transition and final states
# are reported at each applied potential.
Volmer_fcc_ads = {

'IS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.84961609e-02,  1.62926404e+00,  8.97273369e+00],
       [ 2.82450755e+00,  1.62524444e+00,  8.96958219e+00],
       [ 5.63645840e+00,  1.62364654e+00,  8.96585440e+00],
       [ 1.42261485e+00,  4.05364069e+00,  8.98072141e+00],
       [ 4.23080315e+00,  4.06155934e+00,  8.97443405e+00],
       [ 7.03379424e+00,  4.05709525e+00,  8.98151259e+00],
       [ 1.00425641e-02,  8.30749874e-03,  1.13315860e+01],
       [ 2.84595872e+00,  7.11664452e-03,  1.13544705e+01],
       [ 5.63243881e+00, -1.94048918e-04,  1.13690020e+01],
       [ 1.43753230e+00,  2.43425061e+00,  1.13731753e+01],
       [ 4.27136608e+00,  2.46142024e+00,  1.13081403e+01],
       [ 7.00268247e+00,  2.45579316e+00,  1.13651107e+01],
       [ 5.65516988e+00, -1.65068653e-01,  1.49045331e+01],
       [ 5.49650856e+00, -2.03282416e-02,  1.39353808e+01],
       [ 6.11244963e+00,  6.48310352e-01,  1.52032487e+01],
       [ 1.28972904e+00,  2.70877479e+00,  1.49331131e+01],
       [ 1.75026262e+00,  3.55159861e+00,  1.52501025e+01],
       [ 1.56125552e+00,  2.61605957e+00,  1.39884702e+01],
       [ 7.20830004e+00,  2.66278773e+00,  1.46658604e+01],
       [ 6.65694515e+00,  3.52342541e+00,  1.48784885e+01],
       [ 8.26701744e+00,  2.77118710e+00,  1.48564380e+01],
       [ 2.62811215e+00,  3.11948550e-02,  1.57797294e+01],
       [ 3.56993900e+00,  3.88451090e-02,  1.55081302e+01],
       [ 2.27571513e+00,  9.13360406e-01,  1.55330542e+01],
       [ 7.10695319e+00,  2.53877506e+00,  1.36583180e+01],
       [ 1.42430380e+00,  8.06442489e-01,  1.22451372e+01],
       [ 4.25714733e+00,  8.37939574e-01,  1.22467748e+01],
       [ 7.01008721e+00,  8.32048330e-01,  1.22462833e+01],
       [ 2.10629497e-03,  3.25214594e+00,  1.22116029e+01],
       [ 2.88333887e+00,  3.23893381e+00,  1.22411677e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.97333138e-02,  1.62915641e+00,  8.97260182e+00],
       [ 2.83596402e+00,  1.61633784e+00,  8.94990840e+00],
       [ 5.65997022e+00,  1.62055395e+00,  8.97040973e+00],
       [ 1.43143045e+00,  4.04913384e+00,  8.96086012e+00],
       [ 4.24474150e+00,  4.05287435e+00,  8.95979515e+00],
       [ 7.04566764e+00,  4.04391712e+00,  8.98430547e+00],
       [ 2.95394362e-02,  9.36731974e-03,  1.13018044e+01],
       [ 2.86579918e+00, -1.21292042e-03,  1.13260992e+01],
       [ 5.64886437e+00, -7.05279002e-03,  1.13314804e+01],
       [ 1.45922383e+00,  2.42863851e+00,  1.13286799e+01],
       [ 4.29386334e+00,  2.44657914e+00,  1.12805925e+01],
       [ 7.01238077e+00,  2.46287734e+00,  1.14944686e+01],
       [ 5.60433223e+00, -1.99899736e-02,  1.49054105e+01],
       [ 5.41443764e+00, -1.85371353e-02,  1.39375543e+01],
       [ 6.11752328e+00,  8.11994911e-01,  1.50305122e+01],
       [ 1.41745006e+00,  2.64901549e+00,  1.49494532e+01],
       [ 1.86529383e+00,  3.49350661e+00,  1.52498769e+01],
       [ 1.64715298e+00,  2.59020303e+00,  1.39961216e+01],
       [ 7.07650597e+00,  2.56143361e+00,  1.45774440e+01],
       [ 6.62769232e+00,  3.42409809e+00,  1.48365792e+01],
       [ 8.05689088e+00,  2.65917978e+00,  1.47807288e+01],
       [ 2.72089204e+00,  1.03379789e-01,  1.57718548e+01],
       [ 3.66784570e+00,  1.35224337e-01,  1.55079313e+01],
       [ 2.34286775e+00,  9.74004039e-01,  1.55048750e+01],
       [ 6.97669197e+00,  2.55985937e+00,  1.31958201e+01],
       [ 1.45044835e+00,  8.11967387e-01,  1.22204192e+01],
       [ 4.26716938e+00,  8.33868950e-01,  1.22205024e+01],
       [ 7.04912499e+00,  7.39068487e-01,  1.22347094e+01],
       [ 1.11202613e-01,  3.27357579e+00,  1.22055875e+01],
       [ 2.90995298e+00,  3.24051689e+00,  1.22087937e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.54623956e-03,  1.61702778e+00,  8.96837117e+00],
       [ 2.80467157e+00,  1.61782290e+00,  8.96845705e+00],
       [ 5.61395289e+00,  1.61449406e+00,  8.96756901e+00],
       [ 1.40241892e+00,  4.04529295e+00,  8.96805744e+00],
       [ 4.21080450e+00,  4.04898048e+00,  8.96766175e+00],
       [ 7.01555349e+00,  4.04887616e+00,  8.96701014e+00],
       [ 6.52177949e-04,  8.70628734e-04,  1.13339031e+01],
       [ 2.80582507e+00,  1.19677227e-03,  1.13313367e+01],
       [ 5.61445973e+00,  1.41016679e-03,  1.13612653e+01],
       [ 1.40384843e+00,  2.43140791e+00,  1.13655212e+01],
       [ 4.21159300e+00,  2.43153558e+00,  1.13298825e+01],
       [ 7.01869998e+00,  2.43137416e+00,  1.13289668e+01],
       [ 5.63709154e+00,  1.73488681e-01,  1.49434405e+01],
       [ 5.61322528e+00,  1.35106597e-01,  1.39614987e+01],
       [ 6.13334341e+00,  1.01800841e+00,  1.51404298e+01],
       [ 1.46434224e+00,  2.65074079e+00,  1.49350227e+01],
       [ 1.92764397e+00,  3.50980688e+00,  1.51532926e+01],
       [ 1.46241340e+00,  2.61816424e+00,  1.39527173e+01],
       [ 7.01115260e+00,  2.52131036e+00,  1.55553811e+01],
       [ 6.60314451e+00,  3.39113422e+00,  1.53272227e+01],
       [ 7.96451658e+00,  2.60125126e+00,  1.53157254e+01],
       [ 2.76294099e+00,  1.49065953e-01,  1.56268336e+01],
       [ 3.71519496e+00,  2.07961419e-01,  1.53758224e+01],
       [ 2.36759001e+00,  1.01686192e+00,  1.53667450e+01],
       [ 5.61494305e+00,  3.23629128e+00,  1.22421808e+01],
       [ 1.39920301e+00,  8.01456813e-01,  1.22519332e+01],
       [ 4.20211615e+00,  8.15742909e-01,  1.22434408e+01],
       [ 7.03165369e+00,  8.10814324e-01,  1.22444492e+01],
       [-1.37785974e-02,  3.24808458e+00,  1.22452641e+01],
       [ 2.82507108e+00,  3.24505748e+00,  1.22423839e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.89501597e-02,  1.63025811e+00,  8.97107963e+00],
       [ 2.82399797e+00,  1.62458248e+00,  8.96619736e+00],
       [ 5.63791252e+00,  1.62474627e+00,  8.96858942e+00],
       [ 1.42327888e+00,  4.05445309e+00,  8.97579796e+00],
       [ 4.23223044e+00,  4.06141405e+00,  8.97372938e+00],
       [ 7.03306033e+00,  4.05647162e+00,  8.98203868e+00],
       [ 9.73506490e-03,  8.78766278e-03,  1.13313210e+01],
       [ 2.84529914e+00,  5.86160372e-03,  1.13556874e+01],
       [ 5.63147060e+00, -1.58719530e-03,  1.13701241e+01],
       [ 1.43782599e+00,  2.43389132e+00,  1.13766292e+01],
       [ 4.27017014e+00,  2.45992829e+00,  1.13097218e+01],
       [ 7.00184712e+00,  2.45742036e+00,  1.13733069e+01],
       [ 5.65593059e+00, -1.64608462e-01,  1.49049846e+01],
       [ 5.49961844e+00, -1.91459988e-02,  1.39346618e+01],
       [ 6.10836253e+00,  6.51956536e-01,  1.52029747e+01],
       [ 1.29400112e+00,  2.70876609e+00,  1.49311751e+01],
       [ 1.75228010e+00,  3.55288534e+00,  1.52473813e+01],
       [ 1.55974222e+00,  2.61962326e+00,  1.39833602e+01],
       [ 7.20783254e+00,  2.65853327e+00,  1.46666890e+01],
       [ 6.65880657e+00,  3.52117544e+00,  1.48779580e+01],
       [ 8.26443432e+00,  2.77092510e+00,  1.48531868e+01],
       [ 2.62621959e+00,  3.32825878e-02,  1.57836473e+01],
       [ 3.56650249e+00,  3.57167661e-02,  1.55055072e+01],
       [ 2.27189658e+00,  9.13487723e-01,  1.55312702e+01],
       [ 7.10946624e+00,  2.54071500e+00,  1.36548548e+01],
       [ 1.42406082e+00,  8.06862500e-01,  1.22471293e+01],
       [ 4.25691705e+00,  8.38967571e-01,  1.22475762e+01],
       [ 7.01068652e+00,  8.30063023e-01,  1.22471733e+01],
       [ 4.47828277e-03,  3.25325543e+00,  1.22133898e+01],
       [ 2.88384733e+00,  3.23921497e+00,  1.22427022e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.80690677e-02,  1.63288169e+00,  8.98312971e+00],
       [ 2.83496635e+00,  1.61984049e+00,  8.96073755e+00],
       [ 5.65964918e+00,  1.62370340e+00,  8.98018991e+00],
       [ 1.42946437e+00,  4.05100084e+00,  8.97053163e+00],
       [ 4.24395127e+00,  4.05670826e+00,  8.96677698e+00],
       [ 7.04450892e+00,  4.04510387e+00,  8.99274141e+00],
       [ 2.09773488e-02,  8.27990546e-03,  1.13132387e+01],
       [ 2.85801738e+00, -3.30898717e-04,  1.13408677e+01],
       [ 5.64320995e+00, -2.94101471e-03,  1.13469705e+01],
       [ 1.44717569e+00,  2.42941909e+00,  1.13470100e+01],
       [ 4.28500323e+00,  2.45240265e+00,  1.12943732e+01],
       [ 7.00210244e+00,  2.46325465e+00,  1.15434792e+01],
       [ 5.56702748e+00, -4.29915520e-02,  1.48696616e+01],
       [ 5.40634546e+00, -1.46156571e-02,  1.38932364e+01],
       [ 6.07474982e+00,  7.83190503e-01,  1.50361306e+01],
       [ 1.36830639e+00,  2.63437579e+00,  1.49331326e+01],
       [ 1.81703285e+00,  3.47805050e+00,  1.52320986e+01],
       [ 1.58932336e+00,  2.57597486e+00,  1.39750087e+01],
       [ 7.05573419e+00,  2.56679566e+00,  1.46134919e+01],
       [ 6.59438788e+00,  3.42970488e+00,  1.48607164e+01],
       [ 8.04092363e+00,  2.66748913e+00,  1.48166985e+01],
       [ 2.69549466e+00,  7.74032147e-02,  1.57829624e+01],
       [ 3.63346885e+00,  1.05920301e-01,  1.54876693e+01],
       [ 2.30975800e+00,  9.42476428e-01,  1.55109676e+01],
       [ 6.96758242e+00,  2.54874646e+00,  1.32910514e+01],
       [ 1.43940331e+00,  8.08122916e-01,  1.22305853e+01],
       [ 4.25557217e+00,  8.38554562e-01,  1.22299233e+01],
       [ 7.04514660e+00,  7.50175471e-01,  1.22537604e+01],
       [ 9.56578395e-02,  3.27165206e+00,  1.22300995e+01],
       [ 2.90258388e+00,  3.23845384e+00,  1.22259002e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 3.40781559e-03,  1.61773846e+00,  8.96918869e+00],
       [ 2.80570736e+00,  1.61846372e+00,  8.96935567e+00],
       [ 5.61437153e+00,  1.61388530e+00,  8.96912127e+00],
       [ 1.40394746e+00,  4.04507147e+00,  8.96802989e+00],
       [ 4.21316171e+00,  4.04863307e+00,  8.96847764e+00],
       [ 7.01608403e+00,  4.04981634e+00,  8.96790515e+00],
       [ 1.04528194e-03,  8.50951836e-05,  1.13325541e+01],
       [ 2.80706426e+00,  4.72858836e-04,  1.13293922e+01],
       [ 5.61437166e+00,  1.52091160e-04,  1.13677955e+01],
       [ 1.40427284e+00,  2.43221325e+00,  1.13674520e+01],
       [ 4.21249366e+00,  2.43177990e+00,  1.13320769e+01],
       [ 7.01825208e+00,  2.43251796e+00,  1.13300988e+01],
       [ 5.63806895e+00,  1.77283370e-01,  1.49433527e+01],
       [ 5.61468117e+00,  1.39014843e-01,  1.39599771e+01],
       [ 6.13185667e+00,  1.02224570e+00,  1.51432045e+01],
       [ 1.46378988e+00,  2.64724171e+00,  1.49341919e+01],
       [ 1.92755334e+00,  3.50521733e+00,  1.51568575e+01],
       [ 1.46120462e+00,  2.62126308e+00,  1.39501629e+01],
       [ 7.01738254e+00,  2.53499179e+00,  1.55939107e+01],
       [ 6.59671573e+00,  3.38774734e+00,  1.53289227e+01],
       [ 7.96297678e+00,  2.61113627e+00,  1.53212907e+01],
       [ 2.76180102e+00,  1.49019050e-01,  1.56279923e+01],
       [ 3.71342681e+00,  2.06050310e-01,  1.53741945e+01],
       [ 2.36736959e+00,  1.01787483e+00,  1.53680718e+01],
       [ 5.61399813e+00,  3.23025721e+00,  1.22503420e+01],
       [ 1.40352314e+00,  8.00359026e-01,  1.22501191e+01],
       [ 4.19703070e+00,  8.15807777e-01,  1.22466813e+01],
       [ 7.03402379e+00,  8.14283611e-01,  1.22439297e+01],
       [-1.31353419e-02,  3.24483109e+00,  1.22462078e+01],
       [ 2.82535514e+00,  3.24537445e+00,  1.22437645e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.38447633e-02,  1.63820771e+00,  8.97546123e+00],
       [ 2.83203455e+00,  1.63077537e+00,  8.96959437e+00],
       [ 5.64577902e+00,  1.63044853e+00,  8.97403469e+00],
       [ 1.43011297e+00,  4.06052500e+00,  8.97818632e+00],
       [ 4.23961175e+00,  4.06703476e+00,  8.97725083e+00],
       [ 7.03880520e+00,  4.06230101e+00,  8.98638644e+00],
       [ 1.80503857e-02,  1.83031357e-02,  1.13296644e+01],
       [ 2.85434617e+00,  1.62191411e-02,  1.13592070e+01],
       [ 5.63989284e+00,  8.59295148e-03,  1.13821924e+01],
       [ 1.44719861e+00,  2.44408816e+00,  1.13806602e+01],
       [ 4.28002970e+00,  2.47036629e+00,  1.13160295e+01],
       [ 7.00958362e+00,  2.46817755e+00,  1.13928647e+01],
       [ 5.63713763e+00, -1.81542871e-01,  1.48748443e+01],
       [ 5.50142293e+00, -4.37213279e-02,  1.38973722e+01],
       [ 6.08859537e+00,  6.36011995e-01,  1.51717554e+01],
       [ 1.28861651e+00,  2.68456267e+00,  1.48989960e+01],
       [ 1.73594677e+00,  3.52791937e+00,  1.52283354e+01],
       [ 1.54420261e+00,  2.62491414e+00,  1.39433830e+01],
       [ 7.18702776e+00,  2.63309068e+00,  1.46404005e+01],
       [ 6.64533797e+00,  3.49768594e+00,  1.48579535e+01],
       [ 8.23639413e+00,  2.74345074e+00,  1.48281183e+01],
       [ 2.58747079e+00,  3.63699589e-02,  1.57834878e+01],
       [ 3.52280152e+00,  2.70377824e-02,  1.54902863e+01],
       [ 2.23492139e+00,  9.09068292e-01,  1.54992837e+01],
       [ 7.09315012e+00,  2.53324109e+00,  1.36157381e+01],
       [ 1.43064502e+00,  8.15191845e-01,  1.22495002e+01],
       [ 4.26036234e+00,  8.50946232e-01,  1.22544619e+01],
       [ 7.02209891e+00,  8.29593158e-01,  1.22550908e+01],
       [ 2.23618629e-02,  3.26582092e+00,  1.22190747e+01],
       [ 2.89529893e+00,  3.24921398e+00,  1.22482428e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.16745012e-02,  1.63446371e+00,  8.98102313e+00],
       [ 2.82722639e+00,  1.62374327e+00,  8.96080393e+00],
       [ 5.64862070e+00,  1.62823213e+00,  8.97717626e+00],
       [ 1.42240048e+00,  4.05373187e+00,  8.96969043e+00],
       [ 4.23544806e+00,  4.06053897e+00,  8.96498538e+00],
       [ 7.03767240e+00,  4.05120539e+00,  8.98750703e+00],
       [ 8.58952061e-03,  1.21656327e-02,  1.13182353e+01],
       [ 2.84417734e+00,  7.28041786e-03,  1.13372338e+01],
       [ 5.62936432e+00, -1.65436904e-03,  1.13493576e+01],
       [ 1.43553644e+00,  2.43556487e+00,  1.13503905e+01],
       [ 4.27350138e+00,  2.46113092e+00,  1.12981115e+01],
       [ 6.99012442e+00,  2.46724266e+00,  1.15123256e+01],
       [ 5.50118075e+00, -5.64052815e-02,  1.48137053e+01],
       [ 5.37309929e+00, -3.40694903e-02,  1.38287708e+01],
       [ 5.98637592e+00,  7.80108573e-01,  1.49930914e+01],
       [ 1.28729967e+00,  2.63068177e+00,  1.48931081e+01],
       [ 1.74926114e+00,  3.47185326e+00,  1.51819779e+01],
       [ 1.50012983e+00,  2.56648556e+00,  1.39313491e+01],
       [ 6.99881336e+00,  2.57567487e+00,  1.45707870e+01],
       [ 6.52759676e+00,  3.43949567e+00,  1.48115795e+01],
       [ 7.98674793e+00,  2.67214347e+00,  1.47803292e+01],
       [ 2.64114713e+00,  6.12851240e-02,  1.56849383e+01],
       [ 3.57149734e+00,  8.75408944e-02,  1.53619894e+01],
       [ 2.24313701e+00,  9.22479662e-01,  1.54188811e+01],
       [ 6.93466223e+00,  2.55060311e+00,  1.32894453e+01],
       [ 1.42976056e+00,  8.16733289e-01,  1.22262273e+01],
       [ 4.23989489e+00,  8.49690058e-01,  1.22222378e+01],
       [ 7.03338525e+00,  7.57926280e-01,  1.22469873e+01],
       [ 7.67759849e-02,  3.27779543e+00,  1.22240586e+01],
       [ 2.88720562e+00,  3.24953374e+00,  1.22223325e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.63009765e-03,  1.61867804e+00,  8.96876351e+00],
       [ 2.80529571e+00,  1.61875135e+00,  8.96869529e+00],
       [ 5.61499747e+00,  1.61470203e+00,  8.96834415e+00],
       [ 1.40350446e+00,  4.04561770e+00,  8.96847885e+00],
       [ 4.21305582e+00,  4.04955678e+00,  8.96848271e+00],
       [ 7.01592600e+00,  4.04952237e+00,  8.96882579e+00],
       [ 1.19325734e-03,  5.11833719e-04,  1.13353123e+01],
       [ 2.80690182e+00,  8.59983101e-04,  1.13330291e+01],
       [ 5.61445185e+00,  6.91097586e-04,  1.13715057e+01],
       [ 1.40399956e+00,  2.43219510e+00,  1.13715813e+01],
       [ 4.21224492e+00,  2.43160506e+00,  1.13347304e+01],
       [ 7.01837323e+00,  2.43192755e+00,  1.13326378e+01],
       [ 5.63659143e+00,  1.74437034e-01,  1.49372858e+01],
       [ 5.61531226e+00,  1.40104816e-01,  1.39530261e+01],
       [ 6.13115482e+00,  1.01708756e+00,  1.51392029e+01],
       [ 1.46290480e+00,  2.64823088e+00,  1.49294871e+01],
       [ 1.92690223e+00,  3.50525909e+00,  1.51502029e+01],
       [ 1.46167896e+00,  2.62026744e+00,  1.39449473e+01],
       [ 7.01760286e+00,  2.53833438e+00,  1.55982156e+01],
       [ 6.59922656e+00,  3.39247189e+00,  1.53318349e+01],
       [ 7.96224483e+00,  2.61099481e+00,  1.53212179e+01],
       [ 2.76270162e+00,  1.50442098e-01,  1.56284499e+01],
       [ 3.71310257e+00,  2.07074471e-01,  1.53704015e+01],
       [ 2.36691188e+00,  1.01688846e+00,  1.53633777e+01],
       [ 5.61449103e+00,  3.23020626e+00,  1.22510972e+01],
       [ 1.40260365e+00,  7.99679729e-01,  1.22523682e+01],
       [ 4.19729315e+00,  8.16374384e-01,  1.22480354e+01],
       [ 7.03446916e+00,  8.14470521e-01,  1.22466794e+01],
       [-1.41534577e-02,  3.24638702e+00,  1.22482279e+01],
       [ 2.82602468e+00,  3.24620319e+00,  1.22455340e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.24492702e-02,  1.63583641e+00,  8.97769453e+00],
       [ 2.83024177e+00,  1.63143411e+00,  8.97136375e+00],
       [ 5.64439601e+00,  1.63084184e+00,  8.97637564e+00],
       [ 1.42726121e+00,  4.06105486e+00,  8.98136604e+00],
       [ 4.23853181e+00,  4.06775774e+00,  8.98006540e+00],
       [ 7.03742314e+00,  4.06188906e+00,  8.98914695e+00],
       [ 1.42558750e-02,  1.61299223e-02,  1.13388060e+01],
       [ 2.85381508e+00,  1.51865520e-02,  1.13618007e+01],
       [ 5.63916965e+00,  6.61357870e-03,  1.13912446e+01],
       [ 1.44540439e+00,  2.44300741e+00,  1.13813661e+01],
       [ 4.28092881e+00,  2.46883546e+00,  1.13189271e+01],
       [ 7.00887749e+00,  2.46604811e+00,  1.14000520e+01],
       [ 5.64538966e+00, -1.71312863e-01,  1.48811695e+01],
       [ 5.51518099e+00, -4.21168260e-02,  1.39003234e+01],
       [ 6.09677463e+00,  6.49510251e-01,  1.51696773e+01],
       [ 1.29665050e+00,  2.68740041e+00,  1.48973644e+01],
       [ 1.73699142e+00,  3.53370360e+00,  1.52248335e+01],
       [ 1.54371638e+00,  2.63551959e+00,  1.39380697e+01],
       [ 7.18400826e+00,  2.62954462e+00,  1.46312941e+01],
       [ 6.65093730e+00,  3.49469958e+00,  1.48576089e+01],
       [ 8.22774275e+00,  2.73732590e+00,  1.48193286e+01],
       [ 2.60032423e+00,  4.00095315e-02,  1.57932469e+01],
       [ 3.53332340e+00,  3.12442184e-02,  1.54908117e+01],
       [ 2.24195601e+00,  9.07649621e-01,  1.55015685e+01],
       [ 7.08710412e+00,  2.53615629e+00,  1.35973624e+01],
       [ 1.42757379e+00,  8.09673987e-01,  1.22486685e+01],
       [ 4.25423644e+00,  8.48987984e-01,  1.22593444e+01],
       [ 7.02097913e+00,  8.23281054e-01,  1.22658729e+01],
       [ 2.54436287e-02,  3.26742497e+00,  1.22264253e+01],
       [ 2.89276769e+00,  3.24546337e+00,  1.22471270e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.15536300e-02,  1.63668599e+00,  8.98211570e+00],
       [ 2.83083628e+00,  1.62877480e+00,  8.96363643e+00],
       [ 5.64964505e+00,  1.62907809e+00,  8.98290637e+00],
       [ 1.42346927e+00,  4.05810213e+00,  8.97152528e+00],
       [ 4.23851974e+00,  4.06464722e+00,  8.97142200e+00],
       [ 7.03596684e+00,  4.05319003e+00,  8.99259645e+00],
       [ 6.09032320e-03,  1.62550686e-02,  1.13226331e+01],
       [ 2.84694228e+00,  1.28723019e-02,  1.13415143e+01],
       [ 5.63306478e+00,  3.29174775e-03,  1.13680249e+01],
       [ 1.43730589e+00,  2.44063252e+00,  1.13508615e+01],
       [ 4.27650777e+00,  2.46586679e+00,  1.13094968e+01],
       [ 6.99623003e+00,  2.46516158e+00,  1.15113305e+01],
       [ 5.53572427e+00, -8.69691084e-02,  1.48290794e+01],
       [ 5.42095565e+00, -2.86534012e-02,  1.38414959e+01],
       [ 6.01944132e+00,  7.40561393e-01,  1.50426047e+01],
       [ 1.31135379e+00,  2.64412853e+00,  1.48923628e+01],
       [ 1.76021827e+00,  3.48903116e+00,  1.51975734e+01],
       [ 1.53098313e+00,  2.59598920e+00,  1.39298062e+01],
       [ 7.07128295e+00,  2.57818834e+00,  1.45651946e+01],
       [ 6.58775561e+00,  3.43577528e+00,  1.48055815e+01],
       [ 8.06552582e+00,  2.68765245e+00,  1.47735234e+01],
       [ 2.61128429e+00,  5.23522142e-02,  1.57110125e+01],
       [ 3.53751848e+00,  5.38893470e-02,  1.53822903e+01],
       [ 2.22701065e+00,  9.15456978e-01,  1.54301197e+01],
       [ 6.98814626e+00,  2.54293489e+00,  1.33106212e+01],
       [ 1.42796961e+00,  8.20647332e-01,  1.22296205e+01],
       [ 4.24058698e+00,  8.53766810e-01,  1.22366154e+01],
       [ 7.02832546e+00,  7.59318304e-01,  1.22658728e+01],
       [ 8.56846014e-02,  3.29209146e+00,  1.22294770e+01],
       [ 2.88786967e+00,  3.24924187e+00,  1.22233636e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH8',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.81500473e-03,  1.61771870e+00,  8.96944737e+00],
       [ 2.80441681e+00,  1.61835927e+00,  8.96954276e+00],
       [ 5.61408498e+00,  1.61377949e+00,  8.96889437e+00],
       [ 1.40361214e+00,  4.04467667e+00,  8.96949417e+00],
       [ 4.21330613e+00,  4.04919845e+00,  8.96911569e+00],
       [ 7.01603853e+00,  4.04995282e+00,  8.96944263e+00],
       [ 1.13916554e-03,  8.06858983e-04,  1.13339189e+01],
       [ 2.80690887e+00,  1.26671985e-03,  1.13298673e+01],
       [ 5.61437368e+00,  7.64574866e-04,  1.13744035e+01],
       [ 1.40395132e+00,  2.43226620e+00,  1.13761670e+01],
       [ 4.21199012e+00,  2.43176712e+00,  1.13322397e+01],
       [ 7.01810819e+00,  2.43185046e+00,  1.13285976e+01],
       [ 5.63747284e+00,  1.73813193e-01,  1.49379130e+01],
       [ 5.61332566e+00,  1.35510211e-01,  1.39525950e+01],
       [ 6.13055538e+00,  1.01858482e+00,  1.51361620e+01],
       [ 1.46361736e+00,  2.64701554e+00,  1.49290195e+01],
       [ 1.92685072e+00,  3.50421273e+00,  1.51501582e+01],
       [ 1.46014304e+00,  2.61816590e+00,  1.39430022e+01],
       [ 7.02033732e+00,  2.53737919e+00,  1.55965888e+01],
       [ 6.59921353e+00,  3.38667603e+00,  1.53200085e+01],
       [ 7.96205812e+00,  2.60760380e+00,  1.53100799e+01],
       [ 2.76609620e+00,  1.56293123e-01,  1.56418361e+01],
       [ 3.71254814e+00,  2.06011592e-01,  1.53673193e+01],
       [ 2.36760972e+00,  1.01647916e+00,  1.53602876e+01],
       [ 5.61441904e+00,  3.23176352e+00,  1.22500199e+01],
       [ 1.40236760e+00,  8.00118213e-01,  1.22519685e+01],
       [ 4.19782944e+00,  8.16073378e-01,  1.22474645e+01],
       [ 7.03356437e+00,  8.13792668e-01,  1.22452133e+01],
       [-1.40193518e-02,  3.24606511e+00,  1.22482317e+01],
       [ 2.82577306e+00,  3.24580347e+00,  1.22456717e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5]))
}

# Bulk-reference for Volmer step into an fcc site at each applied potential.
Bulk_ref_fcc_ads = {

'CHE_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH7',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 9.06731570e-04,  1.62366902e+00,  8.97133746e+00],
       [ 2.79624861e+00,  1.62322185e+00,  8.97293018e+00],
       [ 5.60891326e+00,  1.62308327e+00,  8.95844399e+00],
       [ 1.39827240e+00,  4.05025356e+00,  8.98548327e+00],
       [ 4.20340418e+00,  4.05794359e+00,  8.97439787e+00],
       [ 7.01542620e+00,  4.05936782e+00,  8.97436091e+00],
       [-1.86803043e-02,  1.08810679e-03,  1.13555969e+01],
       [ 2.81528045e+00,  1.24203476e-04,  1.13543817e+01],
       [ 5.60859323e+00, -1.81602003e-03,  1.13434121e+01],
       [ 1.39738314e+00,  2.43082438e+00,  1.13904034e+01],
       [ 4.24012720e+00,  2.45386143e+00,  1.13076450e+01],
       [ 6.97852489e+00,  2.45736776e+00,  1.13038840e+01],
       [ 5.66144259e+00, -2.59932126e-02,  1.50762129e+01],
       [ 5.67188885e+00, -6.51571235e-02,  1.40955921e+01],
       [ 6.14403223e+00,  8.20317717e-01,  1.52960110e+01],
       [ 1.36076708e+00,  2.49418623e+00,  1.51451749e+01],
       [ 1.82513007e+00,  3.34733096e+00,  1.53799010e+01],
       [ 1.35266561e+00,  2.48967824e+00,  1.41651206e+01],
       [ 6.96434433e+00,  2.34647577e+00,  1.58068983e+01],
       [ 6.55452246e+00,  3.20603534e+00,  1.55402880e+01],
       [ 7.91871514e+00,  2.43018452e+00,  1.55669846e+01],
       [ 2.69776345e+00, -1.30810909e-02,  1.58251080e+01],
       [ 3.63795461e+00,  4.22696992e-02,  1.55401908e+01],
       [ 2.29701366e+00,  8.57644628e-01,  1.55840900e+01],
       [ 1.39823833e+00,  8.06539327e-01,  1.22581037e+01],
       [ 4.24252734e+00,  8.29526787e-01,  1.22385809e+01],
       [ 6.97467875e+00,  8.31527077e-01,  1.22360846e+01],
       [-5.58924595e-02,  3.23623621e+00,  1.22456109e+01],
       [ 2.85378652e+00,  3.23064585e+00,  1.22476366e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'CHE_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH7',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.49789728e-03,  1.62642306e+00,  8.97317382e+00],
       [ 2.79925794e+00,  1.62682834e+00,  8.97401558e+00],
       [ 5.61360137e+00,  1.62679481e+00,  8.95871590e+00],
       [ 1.40256552e+00,  4.05370928e+00,  8.98661636e+00],
       [ 4.20715229e+00,  4.06274436e+00,  8.97426931e+00],
       [ 7.01970601e+00,  4.06269540e+00,  8.97489611e+00],
       [-1.40532754e-02,  4.19556604e-03,  1.13593765e+01],
       [ 2.81982266e+00,  3.98245766e-03,  1.13570065e+01],
       [ 5.61398841e+00,  1.00636528e-03,  1.13460025e+01],
       [ 1.40262221e+00,  2.43466532e+00,  1.13964140e+01],
       [ 4.24440763e+00,  2.45946859e+00,  1.13062884e+01],
       [ 6.98276586e+00,  2.45939836e+00,  1.13058851e+01],
       [ 5.61527510e+00, -1.52561705e-02,  1.50698680e+01],
       [ 5.61912204e+00, -4.94816923e-02,  1.40873912e+01],
       [ 6.10223880e+00,  8.28316742e-01,  1.52888781e+01],
       [ 1.38809151e+00,  2.48281533e+00,  1.50842191e+01],
       [ 1.86671604e+00,  3.32712613e+00,  1.53231835e+01],
       [ 1.38725949e+00,  2.48059193e+00,  1.41013066e+01],
       [ 6.95773058e+00,  2.34926303e+00,  1.57989275e+01],
       [ 6.54569504e+00,  3.20603063e+00,  1.55306268e+01],
       [ 7.90687591e+00,  2.42789397e+00,  1.55404024e+01],
       [ 2.73622417e+00, -3.60448917e-02,  1.58018634e+01],
       [ 3.67897022e+00,  1.49153890e-02,  1.55168571e+01],
       [ 2.33080520e+00,  8.22090616e-01,  1.55312228e+01],
       [ 1.40131017e+00,  8.07846382e-01,  1.22644476e+01],
       [ 4.25257140e+00,  8.36737144e-01,  1.22396759e+01],
       [ 6.97898117e+00,  8.33483125e-01,  1.22398142e+01],
       [-5.30232472e-02,  3.23914850e+00,  1.22495163e+01],
       [ 2.86087765e+00,  3.24172517e+00,  1.22482458e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'CHE_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH7',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 6.30630878e-03,  1.62803419e+00,  8.97725064e+00],
       [ 2.79932080e+00,  1.62804111e+00,  8.97518042e+00],
       [ 5.61363849e+00,  1.62829052e+00,  8.96021066e+00],
       [ 1.40269523e+00,  4.05470824e+00,  8.98513808e+00],
       [ 4.20700367e+00,  4.06385368e+00,  8.97487240e+00],
       [ 7.01983345e+00,  4.06386712e+00,  8.97431598e+00],
       [-1.41349091e-02,  4.47693494e-03,  1.13614374e+01],
       [ 2.81992108e+00,  5.18261502e-03,  1.13585569e+01],
       [ 5.61519650e+00,  1.27078081e-03,  1.13483596e+01],
       [ 1.40327145e+00,  2.43635079e+00,  1.14013796e+01],
       [ 4.24354647e+00,  2.46002244e+00,  1.13073448e+01],
       [ 6.98250666e+00,  2.45899896e+00,  1.13057244e+01],
       [ 5.60406092e+00, -1.99169005e-02,  1.50454155e+01],
       [ 5.61051247e+00, -5.03865500e-02,  1.40621609e+01],
       [ 6.08600112e+00,  8.23606566e-01,  1.52708882e+01],
       [ 1.38865793e+00,  2.47855681e+00,  1.50644326e+01],
       [ 1.86890266e+00,  3.31921853e+00,  1.53076920e+01],
       [ 1.39281406e+00,  2.47873496e+00,  1.40811863e+01],
       [ 6.95160561e+00,  2.34901133e+00,  1.57937125e+01],
       [ 6.53735675e+00,  3.20194300e+00,  1.55164245e+01],
       [ 7.89785340e+00,  2.42479998e+00,  1.55247450e+01],
       [ 2.73966093e+00, -3.00476972e-02,  1.58085915e+01],
       [ 3.68121613e+00,  1.60624274e-02,  1.55168640e+01],
       [ 2.33502245e+00,  8.25365081e-01,  1.55266329e+01],
       [ 1.40178966e+00,  8.07994255e-01,  1.22686789e+01],
       [ 4.25208976e+00,  8.38262590e-01,  1.22436937e+01],
       [ 6.97850900e+00,  8.35366272e-01,  1.22428232e+01],
       [-5.45413536e-02,  3.23986755e+00,  1.22525426e+01],
       [ 2.86319017e+00,  3.24117943e+00,  1.22516018e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'CHE_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH7',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 7.31042249e-03,  1.62896747e+00,  8.97473637e+00],
       [ 2.79858099e+00,  1.62947684e+00,  8.97486300e+00],
       [ 5.61340402e+00,  1.62688160e+00,  8.95879744e+00],
       [ 1.40239305e+00,  4.05459994e+00,  8.98891603e+00],
       [ 4.20831331e+00,  4.06548062e+00,  8.97643869e+00],
       [ 7.01998949e+00,  4.06619424e+00,  8.97685446e+00],
       [-1.29457138e-02,  6.63362809e-03,  1.13598620e+01],
       [ 2.81919090e+00,  5.44307896e-03,  1.13557799e+01],
       [ 5.61304522e+00,  3.21795903e-03,  1.13524346e+01],
       [ 1.40399553e+00,  2.43736879e+00,  1.14062223e+01],
       [ 4.24434055e+00,  2.46056814e+00,  1.13073207e+01],
       [ 6.98468870e+00,  2.46055832e+00,  1.13053096e+01],
       [ 5.58177713e+00, -2.09308978e-02,  1.50304318e+01],
       [ 5.59128921e+00, -5.92215769e-02,  1.40463544e+01],
       [ 6.06824496e+00,  8.21104775e-01,  1.52508886e+01],
       [ 1.39221981e+00,  2.47251774e+00,  1.50492457e+01],
       [ 1.87070119e+00,  3.31568611e+00,  1.52862451e+01],
       [ 1.39456517e+00,  2.46510425e+00,  1.40639794e+01],
       [ 6.95010211e+00,  2.34895033e+00,  1.57852563e+01],
       [ 6.52795855e+00,  3.19296934e+00,  1.54934114e+01],
       [ 7.89140878e+00,  2.42273949e+00,  1.54975223e+01],
       [ 2.74283243e+00, -2.91532224e-02,  1.57988954e+01],
       [ 3.68436460e+00,  1.79590433e-02,  1.55040513e+01],
       [ 2.33932678e+00,  8.23633968e-01,  1.55062967e+01],
       [ 1.40260903e+00,  8.09870709e-01,  1.22666362e+01],
       [ 4.25088298e+00,  8.38480682e-01,  1.22415528e+01],
       [ 6.97836271e+00,  8.36204535e-01,  1.22409900e+01],
       [-5.57526635e-02,  3.24094134e+00,  1.22513499e+01],
       [ 2.86402145e+00,  3.24002960e+00,  1.22505548e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5]))
}

# Heyrovsky reaction from an ontop site. Initial, transition and final states
# are reported at each applied potential.
Heyrovsky = {

'IS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-5.36526757e-03,  1.61273846e+00,  8.96481875e+00],
       [ 2.78528614e+00,  1.64994564e+00,  8.97674425e+00],
       [ 5.58320940e+00,  1.60795486e+00,  8.96489674e+00],
       [ 1.37528779e+00,  4.05739545e+00,  8.98418377e+00],
       [ 4.20093201e+00,  4.06518259e+00,  8.98644489e+00],
       [ 6.99430162e+00,  4.05731545e+00,  8.97376889e+00],
       [-5.84366698e-02, -9.12529056e-03,  1.14205648e+01],
       [ 2.77493356e+00,  4.91388344e-02,  1.12830559e+01],
       [ 5.60366100e+00, -4.01403100e-03,  1.15107954e+01],
       [ 1.34465876e+00,  2.45734581e+00,  1.14215614e+01],
       [ 4.20806122e+00,  2.46380917e+00,  1.13645454e+01],
       [ 6.98426166e+00,  2.39301355e+00,  1.12365784e+01],
       [ 5.18668524e+00,  1.80380355e-01,  1.52409895e+01],
       [ 5.25769820e+00,  1.53850924e-01,  1.42587653e+01],
       [ 5.76336953e+00,  9.35038877e-01,  1.55169491e+01],
       [ 1.65215354e+00,  2.49771256e+00,  1.52423822e+01],
       [ 2.31934754e+00,  4.24099092e+00,  1.57285636e+01],
       [ 1.66191154e+00,  2.49445451e+00,  1.42572612e+01],
       [ 7.23989426e+00,  2.31948826e+00,  1.54429402e+01],
       [ 6.86167363e+00,  3.09158052e+00,  1.59101175e+01],
       [ 9.10204324e+00,  2.46596988e+00,  1.54640717e+01],
       [ 2.78206654e+00,  2.19889571e-01,  1.60057192e+01],
       [ 3.79837205e+00,  2.28299886e-01,  1.56732206e+01],
       [ 2.28658151e+00,  1.06600944e+00,  1.56919354e+01],
       [ 7.10092441e+00,  2.53823126e+00,  1.44967485e+01],
       [ 1.38747983e+00,  8.63335853e-01,  1.22725997e+01],
       [ 4.15213182e+00,  8.84035850e-01,  1.22716150e+01],
       [ 7.02471113e+00,  6.78067785e-02,  1.25382760e+01],
       [ 4.87984767e-01,  3.53698936e+00,  1.24637298e+01],
       [ 2.78642734e+00,  3.21675883e+00,  1.22111964e+01],
       [ 5.03979021e+00,  3.48772450e+00,  1.24791358e+01],
       [ 6.96427527e+00,  2.29671788e+00,  1.27940093e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.02997350e-02,  1.60601838e+00,  8.96664595e+00],
       [ 2.80571033e+00,  1.62564913e+00,  8.96573920e+00],
       [ 5.60532000e+00,  1.59923453e+00,  8.96832069e+00],
       [ 1.40127578e+00,  4.04361873e+00,  8.97255440e+00],
       [ 4.21690570e+00,  4.04544374e+00,  8.97482031e+00],
       [ 7.01404073e+00,  4.04099155e+00,  8.97051864e+00],
       [-2.16693468e-02, -2.51061805e-02,  1.13828369e+01],
       [ 2.80708216e+00,  2.78848868e-02,  1.12822075e+01],
       [ 5.62908447e+00, -2.81893994e-02,  1.14474212e+01],
       [ 1.38538763e+00,  2.44028431e+00,  1.13842121e+01],
       [ 4.22563435e+00,  2.43820034e+00,  1.13496989e+01],
       [ 7.01324411e+00,  2.37306879e+00,  1.12650924e+01],
       [ 5.69745771e+00, -1.19935808e-01,  1.54198183e+01],
       [ 5.56412431e+00, -4.06685437e-02,  1.44529177e+01],
       [ 6.18152447e+00,  7.05495126e-01,  1.56507501e+01],
       [ 1.51344993e+00,  2.45535694e+00,  1.52624347e+01],
       [ 2.35311817e+00,  4.13181567e+00,  1.58516180e+01],
       [ 1.63108692e+00,  2.43793439e+00,  1.42899480e+01],
       [ 7.21843065e+00,  2.42312644e+00,  1.53511133e+01],
       [ 6.79836532e+00,  3.31280348e+00,  1.55356657e+01],
       [ 8.21925101e+00,  2.49653994e+00,  1.54525996e+01],
       [ 2.73393084e+00,  1.33007921e-01,  1.61393269e+01],
       [ 3.69350035e+00,  6.76161391e-02,  1.59396855e+01],
       [ 1.94713317e+00,  1.60808429e+00,  1.55810900e+01],
       [ 7.08515663e+00,  2.35260611e+00,  1.39370574e+01],
       [ 1.39292829e+00,  8.51580656e-01,  1.22525066e+01],
       [ 4.21017657e+00,  8.66928162e-01,  1.22638648e+01],
       [ 7.04425195e+00,  8.10581170e-02,  1.24726340e+01],
       [ 4.84405406e-01,  3.50359478e+00,  1.24108056e+01],
       [ 2.81052756e+00,  3.20853448e+00,  1.22091737e+01],
       [ 5.09722038e+00,  3.47306079e+00,  1.24293183e+01],
       [ 6.98428946e+00,  2.30485299e+00,  1.30431292e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 6.65047378e-03,  1.61727154e+00,  8.96486683e+00],
       [ 2.80390571e+00,  1.61017288e+00,  8.97093997e+00],
       [ 5.61211531e+00,  1.61628305e+00,  8.96378392e+00],
       [ 1.40651508e+00,  4.04472847e+00,  8.97069148e+00],
       [ 4.20603509e+00,  4.04591008e+00,  8.96990897e+00],
       [ 7.01882561e+00,  4.05390146e+00,  8.96294586e+00],
       [ 2.61537261e-03, -2.52353909e-03,  1.13380339e+01],
       [ 2.80512606e+00, -3.92525244e-03,  1.13477539e+01],
       [ 5.61009080e+00,  4.03666022e-05,  1.13351773e+01],
       [ 1.40482309e+00,  2.42814610e+00,  1.13633570e+01],
       [ 4.20435514e+00,  2.42855066e+00,  1.13294073e+01],
       [ 7.01945088e+00,  2.42714059e+00,  1.13078609e+01],
       [ 5.87013003e+00,  4.44437910e-03,  1.61283446e+01],
       [ 6.12833867e+00, -4.61308495e-02,  1.51915935e+01],
       [ 6.30562847e+00,  8.40743922e-01,  1.64498850e+01],
       [ 1.48623479e+00,  2.45955847e+00,  1.56080571e+01],
       [ 2.51481853e+00,  4.11106125e+00,  1.59001049e+01],
       [ 1.27371215e+00,  2.45420415e+00,  1.46560889e+01],
       [ 7.19843140e+00,  2.35126908e+00,  1.70433891e+01],
       [ 6.76714143e+00,  3.20668034e+00,  1.68052614e+01],
       [ 8.08239332e+00,  2.39294941e+00,  1.66127401e+01],
       [ 2.87682309e+00,  1.25980246e-01,  1.61703715e+01],
       [ 3.85367698e+00,  6.80844361e-02,  1.60664975e+01],
       [ 2.00621842e+00,  1.61993976e+00,  1.57572389e+01],
       [ 6.72218750e+00,  2.39588467e+00,  1.47334570e+01],
       [ 1.40230496e+00,  8.03365041e-01,  1.22613671e+01],
       [ 4.21655113e+00,  8.05133938e-01,  1.22511376e+01],
       [ 7.01681627e+00,  7.73074895e-01,  1.22301880e+01],
       [ 7.15553038e-03,  3.24680266e+00,  1.22413081e+01],
       [ 2.81373602e+00,  3.23582037e+00,  1.22645127e+01],
       [ 5.59052052e+00,  3.25249525e+00,  1.22170362e+01],
       [ 6.76632547e+00,  2.42815568e+00,  1.39800441e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-5.34841153e-03,  1.61259051e+00,  8.96478408e+00],
       [ 2.78537833e+00,  1.64974473e+00,  8.97666199e+00],
       [ 5.58313257e+00,  1.60774142e+00,  8.96484815e+00],
       [ 1.37520054e+00,  4.05749173e+00,  8.98412784e+00],
       [ 4.20096906e+00,  4.06520764e+00,  8.98635581e+00],
       [ 6.99425400e+00,  4.05753994e+00,  8.97385907e+00],
       [-5.88875994e-02, -9.21411424e-03,  1.14208701e+01],
       [ 2.77510037e+00,  4.88959307e-02,  1.12830490e+01],
       [ 5.60414784e+00, -4.16731487e-03,  1.15111460e+01],
       [ 1.34480124e+00,  2.45756154e+00,  1.14219584e+01],
       [ 4.20776978e+00,  2.46388291e+00,  1.13646417e+01],
       [ 6.98431247e+00,  2.39318571e+00,  1.12361362e+01],
       [ 5.18647378e+00,  1.80449992e-01,  1.52412017e+01],
       [ 5.25778101e+00,  1.53865398e-01,  1.42581382e+01],
       [ 5.76340376e+00,  9.34782875e-01,  1.55167965e+01],
       [ 1.65204625e+00,  2.49781027e+00,  1.52427705e+01],
       [ 2.31944234e+00,  4.24122879e+00,  1.57283514e+01],
       [ 1.66208936e+00,  2.49456839e+00,  1.42565583e+01],
       [ 7.23946028e+00,  2.31921500e+00,  1.54433358e+01],
       [ 6.86177550e+00,  3.09138668e+00,  1.59105368e+01],
       [ 9.10211820e+00,  2.46590234e+00,  1.54641596e+01],
       [ 2.78227209e+00,  2.19750236e-01,  1.60056018e+01],
       [ 3.79810898e+00,  2.28315032e-01,  1.56730177e+01],
       [ 2.28672518e+00,  1.06562409e+00,  1.56919242e+01],
       [ 7.10091940e+00,  2.53821467e+00,  1.44965173e+01],
       [ 1.38744261e+00,  8.63244627e-01,  1.22724295e+01],
       [ 4.15212950e+00,  8.83952856e-01,  1.22715961e+01],
       [ 7.02473038e+00,  6.78091884e-02,  1.25380875e+01],
       [ 4.87916397e-01,  3.53704646e+00,  1.24635820e+01],
       [ 2.78636066e+00,  3.21683017e+00,  1.22112206e+01],
       [ 5.03979912e+00,  3.48779086e+00,  1.24790164e+01],
       [ 6.96431064e+00,  2.29670129e+00,  1.27948379e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 9.93409480e-03,  1.60658567e+00,  8.96812537e+00],
       [ 2.80589205e+00,  1.62620316e+00,  8.96533206e+00],
       [ 5.60614708e+00,  1.59957844e+00,  8.96989791e+00],
       [ 1.40102166e+00,  4.04396026e+00,  8.97244069e+00],
       [ 4.21751093e+00,  4.04577648e+00,  8.97475234e+00],
       [ 7.01411188e+00,  4.04088213e+00,  8.97218446e+00],
       [-2.10347549e-02, -2.46808161e-02,  1.13840884e+01],
       [ 2.80730926e+00,  2.80923411e-02,  1.12816229e+01],
       [ 5.62881876e+00, -2.75631463e-02,  1.14503032e+01],
       [ 1.38584443e+00,  2.44072474e+00,  1.13855685e+01],
       [ 4.22554659e+00,  2.43868209e+00,  1.13502046e+01],
       [ 7.01332712e+00,  2.37354184e+00,  1.12690264e+01],
       [ 5.69534506e+00, -1.23072210e-01,  1.54208122e+01],
       [ 5.56382001e+00, -4.14059823e-02,  1.44525788e+01],
       [ 6.17946826e+00,  7.01917366e-01,  1.56539709e+01],
       [ 1.51190346e+00,  2.45533550e+00,  1.52625418e+01],
       [ 2.35239302e+00,  4.13141522e+00,  1.58518128e+01],
       [ 1.63092290e+00,  2.43811323e+00,  1.42887598e+01],
       [ 7.22031499e+00,  2.42470555e+00,  1.53393309e+01],
       [ 6.79819489e+00,  3.31336416e+00,  1.55279653e+01],
       [ 8.22198284e+00,  2.49750370e+00,  1.54440430e+01],
       [ 2.73347952e+00,  1.32200561e-01,  1.61409628e+01],
       [ 3.69286774e+00,  6.69099945e-02,  1.59399154e+01],
       [ 1.94562800e+00,  1.60860539e+00,  1.55816122e+01],
       [ 7.08747520e+00,  2.35503987e+00,  1.39351775e+01],
       [ 1.39419637e+00,  8.51456050e-01,  1.22530165e+01],
       [ 4.20896258e+00,  8.67020558e-01,  1.22645247e+01],
       [ 7.04508888e+00,  8.12167866e-02,  1.24734955e+01],
       [ 4.85608444e-01,  3.50448146e+00,  1.24118814e+01],
       [ 2.81068226e+00,  3.20966326e+00,  1.22097401e+01],
       [ 5.09603575e+00,  3.47341208e+00,  1.24306596e+01],
       [ 6.98533802e+00,  2.30617345e+00,  1.30402940e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 6.64304736e-03,  1.61730571e+00,  8.96495887e+00],
       [ 2.80384388e+00,  1.61051027e+00,  8.97073180e+00],
       [ 5.61200315e+00,  1.61650667e+00,  8.96353732e+00],
       [ 1.40642351e+00,  4.04447510e+00,  8.97071408e+00],
       [ 4.20615343e+00,  4.04584601e+00,  8.96942542e+00],
       [ 7.01885072e+00,  4.05346336e+00,  8.96299884e+00],
       [ 2.48118749e-03, -2.38846585e-03,  1.13381408e+01],
       [ 2.80524629e+00, -3.64875773e-03,  1.13478836e+01],
       [ 5.61030886e+00, -4.62200168e-05,  1.13353088e+01],
       [ 1.40479216e+00,  2.42805494e+00,  1.13637430e+01],
       [ 4.20455813e+00,  2.42826538e+00,  1.13300107e+01],
       [ 7.01927942e+00,  2.42737362e+00,  1.13079716e+01],
       [ 5.87010125e+00,  4.64878666e-03,  1.61290847e+01],
       [ 6.12827720e+00, -4.63106620e-02,  1.51910385e+01],
       [ 6.30569359e+00,  8.40825026e-01,  1.64501614e+01],
       [ 1.48604676e+00,  2.46008854e+00,  1.56086386e+01],
       [ 2.51478448e+00,  4.11108812e+00,  1.59004006e+01],
       [ 1.27410696e+00,  2.45389626e+00,  1.46560326e+01],
       [ 7.19840722e+00,  2.35150782e+00,  1.70437425e+01],
       [ 6.76727348e+00,  3.20652930e+00,  1.68053096e+01],
       [ 8.08285075e+00,  2.39288597e+00,  1.66124418e+01],
       [ 2.87707311e+00,  1.26288274e-01,  1.61704502e+01],
       [ 3.85359002e+00,  6.80365205e-02,  1.60667336e+01],
       [ 2.00617764e+00,  1.61981285e+00,  1.57569624e+01],
       [ 6.72211502e+00,  2.39593513e+00,  1.47337206e+01],
       [ 1.40227896e+00,  8.03412431e-01,  1.22613700e+01],
       [ 4.21654457e+00,  8.05227745e-01,  1.22511379e+01],
       [ 7.01679310e+00,  7.73175487e-01,  1.22301336e+01],
       [ 7.17204439e-03,  3.24673963e+00,  1.22411976e+01],
       [ 2.81372128e+00,  3.23584912e+00,  1.22642618e+01],
       [ 5.59046771e+00,  3.25243533e+00,  1.22170163e+01],
       [ 6.76615133e+00,  2.42823525e+00,  1.39800166e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-4.73554052e-03,  1.61235183e+00,  8.96632829e+00],
       [ 2.78576103e+00,  1.64704531e+00,  8.97662417e+00],
       [ 5.58242623e+00,  1.60602774e+00,  8.96634687e+00],
       [ 1.37383800e+00,  4.05692466e+00,  8.98500875e+00],
       [ 4.20156877e+00,  4.06325989e+00,  8.98579676e+00],
       [ 6.99344224e+00,  4.05923154e+00,  8.97627639e+00],
       [-6.02712745e-02, -7.82452123e-03,  1.14242731e+01],
       [ 2.77685520e+00,  4.66808698e-02,  1.12824074e+01],
       [ 5.60527768e+00, -3.74050985e-03,  1.15218330e+01],
       [ 1.34621138e+00,  2.45781413e+00,  1.14343342e+01],
       [ 4.20542905e+00,  2.46310668e+00,  1.13668455e+01],
       [ 6.98428231e+00,  2.39449719e+00,  1.12422581e+01],
       [ 5.18058383e+00,  1.81222870e-01,  1.52311495e+01],
       [ 5.25717998e+00,  1.52701569e-01,  1.42466944e+01],
       [ 5.76319052e+00,  9.30494747e-01,  1.55095247e+01],
       [ 1.65353406e+00,  2.49908713e+00,  1.52362472e+01],
       [ 2.32094580e+00,  4.24225743e+00,  1.57172606e+01],
       [ 1.66425987e+00,  2.49760765e+00,  1.42485932e+01],
       [ 7.23223333e+00,  2.31468317e+00,  1.54449670e+01],
       [ 6.86212256e+00,  3.08912713e+00,  1.59147464e+01],
       [ 9.10441328e+00,  2.46425541e+00,  1.54600598e+01],
       [ 2.78399687e+00,  2.17425488e-01,  1.60044976e+01],
       [ 3.80013439e+00,  2.28957989e-01,  1.56626605e+01],
       [ 2.28878633e+00,  1.06065838e+00,  1.56870816e+01],
       [ 7.10216738e+00,  2.53175695e+00,  1.44956837e+01],
       [ 1.38730405e+00,  8.59995661e-01,  1.22695762e+01],
       [ 4.15102719e+00,  8.82008279e-01,  1.22717666e+01],
       [ 7.02640621e+00,  6.79660276e-02,  1.25418523e+01],
       [ 4.85336341e-01,  3.53916336e+00,  1.24654587e+01],
       [ 2.78551947e+00,  3.22041882e+00,  1.22149769e+01],
       [ 5.03942101e+00,  3.48763816e+00,  1.24796938e+01],
       [ 6.96544132e+00,  2.29830077e+00,  1.28044515e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 8.96780737e-03,  1.60383477e+00,  8.96859757e+00],
       [ 2.80409644e+00,  1.62472016e+00,  8.96561065e+00],
       [ 5.60454181e+00,  1.59674063e+00,  8.97001741e+00],
       [ 1.39880956e+00,  4.04192993e+00,  8.97277432e+00],
       [ 4.21657517e+00,  4.04429291e+00,  8.97475975e+00],
       [ 7.01259328e+00,  4.03914778e+00,  8.97301883e+00],
       [-2.35290166e-02, -2.96152392e-02,  1.13866707e+01],
       [ 2.80514716e+00,  2.54975317e-02,  1.12798696e+01],
       [ 5.62705674e+00, -3.20181475e-02,  1.14559953e+01],
       [ 1.38334718e+00,  2.43750705e+00,  1.13906517e+01],
       [ 4.22429493e+00,  2.43608010e+00,  1.13501608e+01],
       [ 7.01174229e+00,  2.36868553e+00,  1.12657474e+01],
       [ 5.70134514e+00, -1.33451541e-01,  1.53993603e+01],
       [ 5.56440596e+00, -4.94756237e-02,  1.44307548e+01],
       [ 6.18614937e+00,  6.90671086e-01,  1.56310101e+01],
       [ 1.51073049e+00,  2.45323813e+00,  1.52480234e+01],
       [ 2.35176280e+00,  4.13543376e+00,  1.58393481e+01],
       [ 1.63328778e+00,  2.43627800e+00,  1.42737540e+01],
       [ 7.23309599e+00,  2.43276201e+00,  1.52897932e+01],
       [ 6.80675891e+00,  3.31973751e+00,  1.54855010e+01],
       [ 8.23666126e+00,  2.50149946e+00,  1.54041895e+01],
       [ 2.73356961e+00,  1.32660648e-01,  1.61375265e+01],
       [ 3.69137934e+00,  6.95774420e-02,  1.59289732e+01],
       [ 1.94157396e+00,  1.60584860e+00,  1.55689052e+01],
       [ 7.09770073e+00,  2.35645793e+00,  1.39324216e+01],
       [ 1.39489111e+00,  8.47631860e-01,  1.22536821e+01],
       [ 4.20434219e+00,  8.64377185e-01,  1.22639530e+01],
       [ 7.04412836e+00,  6.19819266e-02,  1.24788636e+01],
       [ 4.93002677e-01,  3.50653194e+00,  1.24183425e+01],
       [ 2.80970755e+00,  3.20499674e+00,  1.22093407e+01],
       [ 5.08326103e+00,  3.47287165e+00,  1.24364007e+01],
       [ 6.97777137e+00,  2.29147654e+00,  1.30132053e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 6.54730917e-03,  1.61696653e+00,  8.96571623e+00],
       [ 2.80342011e+00,  1.61250623e+00,  8.96977445e+00],
       [ 5.61123481e+00,  1.61738359e+00,  8.96219279e+00],
       [ 1.40588419e+00,  4.04317959e+00,  8.97118009e+00],
       [ 4.20683872e+00,  4.04573179e+00,  8.96683643e+00],
       [ 7.01903073e+00,  4.05094585e+00,  8.96316333e+00],
       [ 1.88836932e-03, -1.51129988e-03,  1.13394678e+01],
       [ 2.80613511e+00, -2.50485148e-03,  1.13486943e+01],
       [ 5.61144348e+00, -3.90625052e-04,  1.13366756e+01],
       [ 1.40467078e+00,  2.42781188e+00,  1.13680897e+01],
       [ 4.20560603e+00,  2.42696271e+00,  1.13343490e+01],
       [ 7.01830387e+00,  2.42857921e+00,  1.13093431e+01],
       [ 5.87156001e+00,  5.19999817e-03,  1.61298823e+01],
       [ 6.12539008e+00, -4.70404664e-02,  1.51910712e+01],
       [ 6.30657890e+00,  8.41874295e-01,  1.64512489e+01],
       [ 1.48405153e+00,  2.46083439e+00,  1.56099752e+01],
       [ 2.51478745e+00,  4.11277028e+00,  1.59013941e+01],
       [ 1.27848461e+00,  2.45318494e+00,  1.46562017e+01],
       [ 7.20234029e+00,  2.35338952e+00,  1.70480883e+01],
       [ 6.76813208e+00,  3.20569940e+00,  1.68044118e+01],
       [ 8.08355151e+00,  2.39209657e+00,  1.66101981e+01],
       [ 2.87765001e+00,  1.26741878e-01,  1.61742610e+01],
       [ 3.85393600e+00,  6.77819670e-02,  1.60672144e+01],
       [ 2.00491939e+00,  1.62096486e+00,  1.57547797e+01],
       [ 6.72134032e+00,  2.39656824e+00,  1.47343587e+01],
       [ 1.40208584e+00,  8.03711087e-01,  1.22614917e+01],
       [ 4.21651657e+00,  8.05900967e-01,  1.22512818e+01],
       [ 7.01662191e+00,  7.74267790e-01,  1.22298540e+01],
       [ 7.08141625e-03,  3.24620170e+00,  1.22405108e+01],
       [ 2.81365007e+00,  3.23618269e+00,  1.22623051e+01],
       [ 5.59023687e+00,  3.25185397e+00,  1.22171472e+01],
       [ 6.76459901e+00,  2.42878171e+00,  1.39802883e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-4.62914032e-03,  1.61215078e+00,  8.96605378e+00],
       [ 2.78589785e+00,  1.64696107e+00,  8.97649714e+00],
       [ 5.58227259e+00,  1.60586503e+00,  8.96607588e+00],
       [ 1.37382806e+00,  4.05709023e+00,  8.98486198e+00],
       [ 4.20155289e+00,  4.06337893e+00,  8.98570506e+00],
       [ 6.99343147e+00,  4.05945510e+00,  8.97611306e+00],
       [-6.03774946e-02, -7.79520025e-03,  1.14247635e+01],
       [ 2.77696679e+00,  4.66152129e-02,  1.12827130e+01],
       [ 5.60539503e+00, -3.72842926e-03,  1.15225705e+01],
       [ 1.34640732e+00,  2.45777828e+00,  1.14350955e+01],
       [ 4.20521235e+00,  2.46308633e+00,  1.13672872e+01],
       [ 6.98433042e+00,  2.39450782e+00,  1.12419643e+01],
       [ 5.18039639e+00,  1.81281459e-01,  1.52318823e+01],
       [ 5.25726570e+00,  1.52672665e-01,  1.42464226e+01],
       [ 5.76331000e+00,  9.30624612e-01,  1.55096721e+01],
       [ 1.65343515e+00,  2.49898425e+00,  1.52368279e+01],
       [ 2.32087449e+00,  4.24245489e+00,  1.57172025e+01],
       [ 1.66436916e+00,  2.49756962e+00,  1.42482508e+01],
       [ 7.23182138e+00,  2.31503435e+00,  1.54450452e+01],
       [ 6.86209870e+00,  3.08940425e+00,  1.59150946e+01],
       [ 9.10418615e+00,  2.46426172e+00,  1.54602077e+01],
       [ 2.78383429e+00,  2.17464647e-01,  1.60053475e+01],
       [ 3.80035898e+00,  2.29007020e-01,  1.56625363e+01],
       [ 2.28864158e+00,  1.06072144e+00,  1.56868595e+01],
       [ 7.10238204e+00,  2.53149169e+00,  1.44947490e+01],
       [ 1.38730893e+00,  8.60124806e-01,  1.22693396e+01],
       [ 4.15105284e+00,  8.82041306e-01,  1.22715966e+01],
       [ 7.02631451e+00,  6.79287756e-02,  1.25414388e+01],
       [ 4.85474772e-01,  3.53907171e+00,  1.24651230e+01],
       [ 2.78550344e+00,  3.22033272e+00,  1.22146628e+01],
       [ 5.03939934e+00,  3.48771658e+00,  1.24794450e+01],
       [ 6.96542907e+00,  2.29825389e+00,  1.28048840e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.02523844e-02,  1.60795743e+00,  8.96897005e+00],
       [ 2.80560165e+00,  1.62814711e+00,  8.96530160e+00],
       [ 5.60640805e+00,  1.60043307e+00,  8.97089542e+00],
       [ 1.40045603e+00,  4.04499908e+00,  8.97245277e+00],
       [ 4.21848533e+00,  4.04719833e+00,  8.97483538e+00],
       [ 7.01374820e+00,  4.04288888e+00,  8.97338274e+00],
       [-2.09314512e-02, -2.26901674e-02,  1.13847537e+01],
       [ 2.80773426e+00,  3.02037528e-02,  1.12797543e+01],
       [ 5.62907781e+00, -2.54497265e-02,  1.14587924e+01],
       [ 1.38604394e+00,  2.44284231e+00,  1.13921836e+01],
       [ 4.22583000e+00,  2.44039512e+00,  1.13511305e+01],
       [ 7.01349475e+00,  2.37564769e+00,  1.12716683e+01],
       [ 5.70112659e+00, -1.34804694e-01,  1.53995024e+01],
       [ 5.56602108e+00, -4.06145830e-02,  1.44312198e+01],
       [ 6.17861622e+00,  6.89840994e-01,  1.56429785e+01],
       [ 1.50747407e+00,  2.45794487e+00,  1.52544698e+01],
       [ 2.35060896e+00,  4.13324478e+00,  1.58523994e+01],
       [ 1.63080364e+00,  2.43857193e+00,  1.42795246e+01],
       [ 7.23077595e+00,  2.43810035e+00,  1.52861710e+01],
       [ 6.80068012e+00,  3.32330772e+00,  1.54855054e+01],
       [ 8.23434350e+00,  2.50626036e+00,  1.54085701e+01],
       [ 2.73359341e+00,  1.32261294e-01,  1.61439096e+01],
       [ 3.69114281e+00,  6.62581650e-02,  1.59354812e+01],
       [ 1.93721976e+00,  1.61166668e+00,  1.55775546e+01],
       [ 7.10349513e+00,  2.36488077e+00,  1.39443109e+01],
       [ 1.39977767e+00,  8.51124520e-01,  1.22536450e+01],
       [ 4.20158255e+00,  8.66560316e-01,  1.22655504e+01],
       [ 7.04798868e+00,  8.05346948e-02,  1.24777882e+01],
       [ 4.89503378e-01,  3.50982185e+00,  1.24155685e+01],
       [ 2.81139611e+00,  3.21660969e+00,  1.22103889e+01],
       [ 5.09404963e+00,  3.47218791e+00,  1.24343912e+01],
       [ 6.98964230e+00,  2.30510890e+00,  1.30149706e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 6.54755161e-03,  1.61696459e+00,  8.96574154e+00],
       [ 2.80342079e+00,  1.61250802e+00,  8.96980098e+00],
       [ 5.61123482e+00,  1.61738220e+00,  8.96221777e+00],
       [ 1.40588239e+00,  4.04318137e+00,  8.97120556e+00],
       [ 4.20684002e+00,  4.04573208e+00,  8.96686126e+00],
       [ 7.01903130e+00,  4.05094694e+00,  8.96318613e+00],
       [ 1.88737404e-03, -1.50956078e-03,  1.13395538e+01],
       [ 2.80613655e+00, -2.50670543e-03,  1.13487738e+01],
       [ 5.61144609e+00, -3.85058774e-04,  1.13367856e+01],
       [ 1.40466885e+00,  2.42780772e+00,  1.13682432e+01],
       [ 4.20560800e+00,  2.42695799e+00,  1.13344273e+01],
       [ 7.01829514e+00,  2.42858040e+00,  1.13094393e+01],
       [ 5.87157684e+00,  5.20584659e-03,  1.61298955e+01],
       [ 6.12538688e+00, -4.70432937e-02,  1.51912168e+01],
       [ 6.30657112e+00,  8.41871457e-01,  1.64511968e+01],
       [ 1.48405661e+00,  2.46080036e+00,  1.56102620e+01],
       [ 2.51477822e+00,  4.11275332e+00,  1.59013917e+01],
       [ 1.27849603e+00,  2.45319242e+00,  1.46556324e+01],
       [ 7.20237125e+00,  2.35341144e+00,  1.70482429e+01],
       [ 6.76813319e+00,  3.20567710e+00,  1.68043550e+01],
       [ 8.08354179e+00,  2.39209154e+00,  1.66101424e+01],
       [ 2.87766367e+00,  1.26727581e-01,  1.61742796e+01],
       [ 3.85393592e+00,  6.77824540e-02,  1.60671988e+01],
       [ 2.00489084e+00,  1.62101478e+00,  1.57548003e+01],
       [ 6.72132615e+00,  2.39657702e+00,  1.47344522e+01],
       [ 1.40208784e+00,  8.03710176e-01,  1.22614791e+01],
       [ 4.21651509e+00,  8.05902317e-01,  1.22512635e+01],
       [ 7.01662060e+00,  7.74282697e-01,  1.22298466e+01],
       [ 7.06854400e-03,  3.24619489e+00,  1.22405076e+01],
       [ 2.81365289e+00,  3.23618504e+00,  1.22622886e+01],
       [ 5.59024913e+00,  3.25184361e+00,  1.22171340e+01],
       [ 6.76460121e+00,  2.42877798e+00,  1.39801212e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5]))
}

# Tafel reaction between top and hollow adsorbed hydrogen atoms. Initial,
# transition and final states are reported at each applied potential.
Tafel_top_hollow = {

'IS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.07677862e-03,  1.61463839e+00,  8.95426077e+00],
       [ 2.81633024e+00,  1.62828142e+00,  8.98500526e+00],
       [ 5.60139393e+00,  1.62147912e+00,  8.98604083e+00],
       [ 1.40564718e+00,  4.04365405e+00,  8.95763014e+00],
       [ 4.21383974e+00,  4.04604492e+00,  8.98989310e+00],
       [ 7.01201653e+00,  4.04794692e+00,  8.95819496e+00],
       [-2.64038845e-05,  3.01017651e-03,  1.13104845e+01],
       [ 2.80255704e+00, -3.76239120e-04,  1.13086491e+01],
       [ 5.61941190e+00, -2.27498777e-03,  1.13540143e+01],
       [ 1.40055827e+00,  2.42944709e+00,  1.13450083e+01],
       [ 4.21387100e+00,  2.42606081e+00,  1.15101204e+01],
       [ 7.02136858e+00,  2.42953644e+00,  1.13080632e+01],
       [ 5.59111763e+00,  2.83043490e-02,  1.50558688e+01],
       [ 5.60613614e+00,  1.93376910e-02,  1.40751198e+01],
       [ 6.07793736e+00,  8.68711898e-01,  1.52930782e+01],
       [ 1.36311601e+00,  2.47312169e+00,  1.50938310e+01],
       [ 1.84123399e+00,  3.32091994e+00,  1.53218008e+01],
       [ 1.35471366e+00,  2.46003542e+00,  1.41137918e+01],
       [ 6.92871692e+00,  2.37374782e+00,  1.57657186e+01],
       [ 6.52446853e+00,  3.23298454e+00,  1.54947176e+01],
       [ 7.88266464e+00,  2.44677521e+00,  1.55242854e+01],
       [ 2.72279603e+00, -3.80118599e-02,  1.57806207e+01],
       [ 3.66847322e+00,  2.64467500e-02,  1.55079501e+01],
       [ 2.31681132e+00,  8.26261066e-01,  1.55306333e+01],
       [ 1.38147275e+00,  8.03912674e-01,  1.22355625e+01],
       [ 4.18996208e+00,  6.00499546e-01,  1.22713030e+01],
       [ 7.04852393e+00,  8.23724895e-01,  1.22354622e+01],
       [-2.36726235e-04,  3.24464840e+00,  1.22284997e+01],
       [ 2.62339215e+00,  3.35198077e+00,  1.22683957e+01],
       [ 5.85423137e+00,  3.34666711e+00,  1.22868273e+01],
       [ 4.22868390e+00,  2.43664553e+00,  1.30661290e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 3.16686613e-02,  1.62406914e+00,  8.96566283e+00],
       [ 2.82028971e+00,  1.63692710e+00,  8.98821118e+00],
       [ 5.63857620e+00,  1.63030480e+00,  8.94043253e+00],
       [ 1.42778904e+00,  4.05846495e+00,  8.98795339e+00],
       [ 4.22460675e+00,  4.06941876e+00,  8.96709714e+00],
       [ 7.04727618e+00,  4.05929445e+00,  8.96967733e+00],
       [-6.12753191e-05,  1.62520463e-02,  1.13357967e+01],
       [ 2.84300311e+00,  4.69726095e-03,  1.13908063e+01],
       [ 5.62466011e+00, -4.36965943e-03,  1.13212916e+01],
       [ 1.41966746e+00,  2.45067736e+00,  1.14014331e+01],
       [ 4.26110737e+00,  2.43847180e+00,  1.12791803e+01],
       [ 7.01094060e+00,  2.44916824e+00,  1.12861809e+01],
       [ 5.64152160e+00,  1.73389482e-02,  1.51099518e+01],
       [ 5.68911540e+00, -3.86088985e-03,  1.41312374e+01],
       [ 6.11703798e+00,  8.62966296e-01,  1.53565117e+01],
       [ 1.35538924e+00,  2.45923864e+00,  1.51008112e+01],
       [ 1.83423376e+00,  3.30536954e+00,  1.53277055e+01],
       [ 1.33270839e+00,  2.45062292e+00,  1.41200593e+01],
       [ 6.95078785e+00,  2.35740135e+00,  1.58205590e+01],
       [ 6.54714255e+00,  3.22572573e+00,  1.55784924e+01],
       [ 7.90084970e+00,  2.43194947e+00,  1.55617690e+01],
       [ 2.72312380e+00, -3.58896537e-02,  1.57880640e+01],
       [ 3.67112729e+00,  3.43189613e-02,  1.55290492e+01],
       [ 2.31328854e+00,  8.25435585e-01,  1.55326532e+01],
       [ 1.44095336e+00,  8.66433969e-01,  1.22684595e+01],
       [ 4.28165796e+00,  1.13725622e-01,  1.24133830e+01],
       [ 6.99514945e+00,  8.32954645e-01,  1.22305301e+01],
       [-9.39637885e-02,  3.24413930e+00,  1.22341449e+01],
       [ 2.35045554e+00,  3.51616248e+00,  1.24045450e+01],
       [ 4.84396998e+00,  2.57723778e+00,  1.32101317e+01],
       [ 4.08771131e+00,  2.34238732e+00,  1.33071132e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.55585648e-03,  1.62140390e+00,  8.96888565e+00],
       [ 2.79423897e+00,  1.62409785e+00,  8.96894068e+00],
       [ 5.61026713e+00,  1.62177335e+00,  8.95533361e+00],
       [ 1.39557827e+00,  4.05151060e+00,  8.98456805e+00],
       [ 4.20284028e+00,  4.06026025e+00,  8.97078198e+00],
       [ 7.01592262e+00,  4.05945786e+00,  8.97345050e+00],
       [-2.07409464e-02,  1.87985314e-03,  1.13583469e+01],
       [ 2.81174383e+00, -2.54885485e-04,  1.13476445e+01],
       [ 5.60981733e+00, -2.19882118e-03,  1.13397779e+01],
       [ 1.39352994e+00,  2.43027596e+00,  1.13824627e+01],
       [ 4.23873230e+00,  2.45584822e+00,  1.12939945e+01],
       [ 6.97711790e+00,  2.45571987e+00,  1.13031617e+01],
       [ 5.71275446e+00, -2.65454169e-02,  1.50928156e+01],
       [ 5.67962053e+00, -8.69312676e-02,  1.41154345e+01],
       [ 6.16969177e+00,  8.43018725e-01,  1.52800117e+01],
       [ 1.32015622e+00,  2.51407412e+00,  1.51585307e+01],
       [ 1.79772870e+00,  3.36101222e+00,  1.53880979e+01],
       [ 1.34640612e+00,  2.48499995e+00,  1.41794680e+01],
       [ 6.97436999e+00,  2.34412288e+00,  1.58104506e+01],
       [ 6.55745967e+00,  3.19634996e+00,  1.55316333e+01],
       [ 7.92859252e+00,  2.43391077e+00,  1.55640878e+01],
       [ 2.69099343e+00, -1.10221925e-02,  1.58480854e+01],
       [ 3.62099642e+00,  4.46216082e-02,  1.55339940e+01],
       [ 2.27580607e+00,  8.52493153e-01,  1.56111161e+01],
       [ 1.39990964e+00,  8.08046893e-01,  1.22574407e+01],
       [ 4.23817266e+00,  8.07022170e-01,  1.22257839e+01],
       [ 6.97228398e+00,  8.31868489e-01,  1.22370076e+01],
       [-5.87709150e-02,  3.23686608e+00,  1.22477750e+01],
       [ 2.83510133e+00,  3.23592919e+00,  1.22284923e+01],
       [ 4.27154502e+00,  2.53469503e+00,  1.41564198e+01],
       [ 4.18387096e+00,  2.56847853e+00,  1.49031990e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.10500261e-03,  1.61465642e+00,  8.95417305e+00],
       [ 2.81637913e+00,  1.62829053e+00,  8.98499979e+00],
       [ 5.60140259e+00,  1.62151559e+00,  8.98605589e+00],
       [ 1.40565800e+00,  4.04367082e+00,  8.95756552e+00],
       [ 4.21379372e+00,  4.04599265e+00,  8.98969764e+00],
       [ 7.01201117e+00,  4.04794663e+00,  8.95817782e+00],
       [ 1.09742880e-05,  3.02287018e-03,  1.13105991e+01],
       [ 2.80259622e+00, -3.99118699e-04,  1.13087393e+01],
       [ 5.61937603e+00, -2.25170252e-03,  1.13542754e+01],
       [ 1.40051246e+00,  2.42940918e+00,  1.13453325e+01],
       [ 4.21379314e+00,  2.42610025e+00,  1.15099740e+01],
       [ 7.02139169e+00,  2.42957048e+00,  1.13081789e+01],
       [ 5.59114899e+00,  2.85374965e-02,  1.50564492e+01],
       [ 5.60598349e+00,  1.92310244e-02,  1.40745754e+01],
       [ 6.07782950e+00,  8.68735138e-01,  1.52933400e+01],
       [ 1.36304367e+00,  2.47319870e+00,  1.50943296e+01],
       [ 1.84114594e+00,  3.32066749e+00,  1.53217414e+01],
       [ 1.35470332e+00,  2.45994821e+00,  1.41130979e+01],
       [ 6.92874554e+00,  2.37369610e+00,  1.57658750e+01],
       [ 6.52443732e+00,  3.23298474e+00,  1.54949648e+01],
       [ 7.88286527e+00,  2.44676305e+00,  1.55241100e+01],
       [ 2.72272408e+00, -3.78774858e-02,  1.57808982e+01],
       [ 3.66855292e+00,  2.65061505e-02,  1.55079180e+01],
       [ 2.31663489e+00,  8.26356033e-01,  1.55304936e+01],
       [ 1.38152707e+00,  8.03980979e-01,  1.22355663e+01],
       [ 4.18996290e+00,  6.00545038e-01,  1.22713097e+01],
       [ 7.04848728e+00,  8.23757491e-01,  1.22354356e+01],
       [-2.66600411e-04,  3.24456202e+00,  1.22284163e+01],
       [ 2.62339693e+00,  3.35198550e+00,  1.22684084e+01],
       [ 5.85425068e+00,  3.34663145e+00,  1.22867845e+01],
       [ 4.22863439e+00,  2.43664755e+00,  1.30661801e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 3.16654715e-02,  1.62429921e+00,  8.96562401e+00],
       [ 2.82037038e+00,  1.63678985e+00,  8.98822985e+00],
       [ 5.63853357e+00,  1.63038984e+00,  8.94069111e+00],
       [ 1.42770450e+00,  4.05832874e+00,  8.98791525e+00],
       [ 4.22459288e+00,  4.06925734e+00,  8.96704240e+00],
       [ 7.04724835e+00,  4.05948780e+00,  8.96974923e+00],
       [-4.71069300e-05,  1.60741684e-02,  1.13357960e+01],
       [ 2.84306106e+00,  4.81380523e-03,  1.13907537e+01],
       [ 5.62453341e+00, -4.11546521e-03,  1.13215124e+01],
       [ 1.41984039e+00,  2.45073781e+00,  1.14017257e+01],
       [ 4.26118934e+00,  2.43817345e+00,  1.12790568e+01],
       [ 7.01046900e+00,  2.44903560e+00,  1.12863787e+01],
       [ 5.64150786e+00,  1.74482837e-02,  1.51106473e+01],
       [ 5.68906962e+00, -3.84283681e-03,  1.41304590e+01],
       [ 6.11691456e+00,  8.62805091e-01,  1.53563230e+01],
       [ 1.35546023e+00,  2.45930549e+00,  1.51013249e+01],
       [ 1.83420310e+00,  3.30532649e+00,  1.53276932e+01],
       [ 1.33264065e+00,  2.45056514e+00,  1.41192347e+01],
       [ 6.95080193e+00,  2.35756308e+00,  1.58208577e+01],
       [ 6.54697807e+00,  3.22573569e+00,  1.55783264e+01],
       [ 7.90089973e+00,  2.43187838e+00,  1.55616507e+01],
       [ 2.72310832e+00, -3.59118667e-02,  1.57882193e+01],
       [ 3.67116709e+00,  3.42772726e-02,  1.55289517e+01],
       [ 2.31325427e+00,  8.25443443e-01,  1.55325991e+01],
       [ 1.44084162e+00,  8.66475547e-01,  1.22685146e+01],
       [ 4.28167244e+00,  1.13734669e-01,  1.24132659e+01],
       [ 6.99508273e+00,  8.33015943e-01,  1.22304776e+01],
       [-9.40127416e-02,  3.24418643e+00,  1.22341044e+01],
       [ 2.35042109e+00,  3.51616716e+00,  1.24044245e+01],
       [ 4.84368793e+00,  2.57702683e+00,  1.32096510e+01],
       [ 4.08825993e+00,  2.34241161e+00,  1.33066574e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 7.14671105e-04,  1.62197532e+00,  8.97267947e+00],
       [ 2.79404361e+00,  1.62441980e+00,  8.96868195e+00],
       [ 5.61133436e+00,  1.62333383e+00,  8.95374086e+00],
       [ 1.39587860e+00,  4.04974313e+00,  8.98030151e+00],
       [ 4.20315996e+00,  4.06010678e+00,  8.96926058e+00],
       [ 7.01612360e+00,  4.05769834e+00,  8.97084529e+00],
       [-2.00288297e-02,  1.17513054e-03,  1.13569752e+01],
       [ 2.81276809e+00,  9.19163754e-04,  1.13444244e+01],
       [ 5.61070151e+00, -1.78849352e-03,  1.13364390e+01],
       [ 1.39494635e+00,  2.43065184e+00,  1.13886085e+01],
       [ 4.23923937e+00,  2.45461282e+00,  1.12938645e+01],
       [ 6.97682552e+00,  2.45463767e+00,  1.13036453e+01],
       [ 5.71493665e+00, -2.85993867e-02,  1.50930537e+01],
       [ 5.67419198e+00, -7.69895989e-02,  1.41145690e+01],
       [ 6.17654025e+00,  8.36581183e-01,  1.52846141e+01],
       [ 1.32176613e+00,  2.50605554e+00,  1.51560460e+01],
       [ 1.79927121e+00,  3.35158469e+00,  1.53876343e+01],
       [ 1.34381636e+00,  2.48261165e+00,  1.41757155e+01],
       [ 6.97431016e+00,  2.34872812e+00,  1.58106974e+01],
       [ 6.56146953e+00,  3.20435014e+00,  1.55345896e+01],
       [ 7.92902943e+00,  2.43614872e+00,  1.55677779e+01],
       [ 2.68799427e+00, -2.15351371e-03,  1.58496552e+01],
       [ 3.61855961e+00,  4.28383807e-02,  1.55368494e+01],
       [ 2.28131771e+00,  8.61656032e-01,  1.55972260e+01],
       [ 1.40008509e+00,  8.08475061e-01,  1.22599042e+01],
       [ 4.23885056e+00,  8.08465966e-01,  1.22295853e+01],
       [ 6.97296412e+00,  8.32671896e-01,  1.22392610e+01],
       [-5.76330046e-02,  3.23716970e+00,  1.22460172e+01],
       [ 2.83549010e+00,  3.23463317e+00,  1.22279324e+01],
       [ 4.26940466e+00,  2.52785307e+00,  1.41551989e+01],
       [ 4.18739518e+00,  2.55552233e+00,  1.49031372e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.49540750e-03,  1.61485382e+00,  8.95318745e+00],
       [ 2.81670720e+00,  1.62837684e+00,  8.98503943e+00],
       [ 5.60165649e+00,  1.62176905e+00,  8.98630076e+00],
       [ 1.40571730e+00,  4.04373917e+00,  8.95692112e+00],
       [ 4.21337886e+00,  4.04554971e+00,  8.98753648e+00],
       [ 7.01195958e+00,  4.04802447e+00,  8.95801390e+00],
       [ 3.72546026e-04,  3.22465743e-03,  1.13120127e+01],
       [ 2.80295822e+00, -7.51848088e-04,  1.13096580e+01],
       [ 5.61904540e+00, -1.95185999e-03,  1.13590688e+01],
       [ 1.40003171e+00,  2.42902045e+00,  1.13508290e+01],
       [ 4.21299619e+00,  2.42643702e+00,  1.15080256e+01],
       [ 7.02154174e+00,  2.42991247e+00,  1.13093728e+01],
       [ 5.59075114e+00,  2.92641489e-02,  1.50558570e+01],
       [ 5.60363900e+00,  1.81838743e-02,  1.40734147e+01],
       [ 6.07669402e+00,  8.69754366e-01,  1.52954573e+01],
       [ 1.36090453e+00,  2.47215878e+00,  1.50916207e+01],
       [ 1.84058183e+00,  3.31844077e+00,  1.53204792e+01],
       [ 1.35463343e+00,  2.45849590e+00,  1.41099354e+01],
       [ 6.93114179e+00,  2.37502483e+00,  1.57703577e+01],
       [ 6.52429416e+00,  3.23241167e+00,  1.54969870e+01],
       [ 7.88371613e+00,  2.44655735e+00,  1.55208372e+01],
       [ 2.72380505e+00, -3.40193785e-02,  1.57868352e+01],
       [ 3.66784606e+00,  2.73657534e-02,  1.55065398e+01],
       [ 2.31481825e+00,  8.26723758e-01,  1.55275518e+01],
       [ 1.38224311e+00,  8.04830563e-01,  1.22356383e+01],
       [ 4.18984508e+00,  6.00387064e-01,  1.22710854e+01],
       [ 7.04805012e+00,  8.24237029e-01,  1.22350035e+01],
       [-7.38218996e-04,  3.24357456e+00,  1.22274666e+01],
       [ 2.62287109e+00,  3.35253921e+00,  1.22682891e+01],
       [ 5.85516788e+00,  3.34640593e+00,  1.22860165e+01],
       [ 4.22800041e+00,  2.43677719e+00,  1.30660733e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 3.04972769e-02,  1.62939243e+00,  8.97099947e+00],
       [ 2.81689512e+00,  1.63111271e+00,  8.98412571e+00],
       [ 5.64154303e+00,  1.63024095e+00,  8.93966413e+00],
       [ 1.42546470e+00,  4.05564047e+00,  8.98879591e+00],
       [ 4.22458723e+00,  4.06749580e+00,  8.96410768e+00],
       [ 7.04610739e+00,  4.06439550e+00,  8.97388158e+00],
       [-1.17258072e-03,  1.67094974e-02,  1.13345928e+01],
       [ 2.84068690e+00,  3.45026823e-03,  1.13870549e+01],
       [ 5.62203980e+00, -7.26480196e-03,  1.13272413e+01],
       [ 1.41727532e+00,  2.44958778e+00,  1.14113418e+01],
       [ 4.25671261e+00,  2.43633686e+00,  1.12686645e+01],
       [ 7.00708026e+00,  2.44857349e+00,  1.12941684e+01],
       [ 5.63902194e+00,  1.56745060e-02,  1.51019526e+01],
       [ 5.68046150e+00, -3.48584853e-03,  1.41207087e+01],
       [ 6.11414814e+00,  8.60771339e-01,  1.53480544e+01],
       [ 1.35449220e+00,  2.45661976e+00,  1.50921828e+01],
       [ 1.83219506e+00,  3.30240184e+00,  1.53201056e+01],
       [ 1.33439725e+00,  2.44902061e+00,  1.41087589e+01],
       [ 6.95238229e+00,  2.36397705e+00,  1.58277289e+01],
       [ 6.54526913e+00,  3.22550366e+00,  1.55668909e+01],
       [ 7.89842863e+00,  2.43285236e+00,  1.55525871e+01],
       [ 2.72888619e+00, -3.36364000e-02,  1.58003840e+01],
       [ 3.67177127e+00,  3.15410029e-02,  1.55214555e+01],
       [ 2.31396660e+00,  8.20649210e-01,  1.55284015e+01],
       [ 1.43455626e+00,  8.65523720e-01,  1.22730053e+01],
       [ 4.27840794e+00,  1.12515845e-01,  1.24168264e+01],
       [ 6.99316818e+00,  8.36592267e-01,  1.22330434e+01],
       [-9.81584102e-02,  3.24603027e+00,  1.22352873e+01],
       [ 2.35191357e+00,  3.52021107e+00,  1.24071819e+01],
       [ 4.85295230e+00,  2.57342932e+00,  1.31825331e+01],
       [ 4.09891039e+00,  2.33762328e+00,  1.32873233e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-3.37839296e-04,  1.62299077e+00,  8.97095266e+00],
       [ 2.79403935e+00,  1.62483692e+00,  8.96893772e+00],
       [ 5.61046306e+00,  1.62376052e+00,  8.95469930e+00],
       [ 1.39553555e+00,  4.05002390e+00,  8.98132223e+00],
       [ 4.20307275e+00,  4.05953081e+00,  8.96969720e+00],
       [ 7.01556563e+00,  4.05730778e+00,  8.97199537e+00],
       [-2.06047938e-02,  4.21358629e-05,  1.13591334e+01],
       [ 2.81294964e+00,  2.53829792e-03,  1.13466763e+01],
       [ 5.60943105e+00, -2.54509987e-03,  1.13398897e+01],
       [ 1.39287155e+00,  2.43026876e+00,  1.13866755e+01],
       [ 4.23968696e+00,  2.45379145e+00,  1.12942128e+01],
       [ 6.97737060e+00,  2.45663574e+00,  1.13038093e+01],
       [ 5.71408192e+00, -2.86332776e-02,  1.50936156e+01],
       [ 5.67753672e+00, -8.31314573e-02,  1.41142885e+01],
       [ 6.17142584e+00,  8.40083792e-01,  1.52791899e+01],
       [ 1.31957510e+00,  2.51120025e+00,  1.51578232e+01],
       [ 1.79863692e+00,  3.35742379e+00,  1.53866088e+01],
       [ 1.34585903e+00,  2.48400267e+00,  1.41766405e+01],
       [ 6.97494317e+00,  2.34666189e+00,  1.58128697e+01],
       [ 6.55896493e+00,  3.19870451e+00,  1.55311317e+01],
       [ 7.92827830e+00,  2.43483938e+00,  1.55640451e+01],
       [ 2.69050205e+00, -7.26353610e-03,  1.58520747e+01],
       [ 3.61861152e+00,  4.43648235e-02,  1.55325699e+01],
       [ 2.27898486e+00,  8.55170957e-01,  1.56048354e+01],
       [ 1.40000253e+00,  8.08353155e-01,  1.22585389e+01],
       [ 4.23829945e+00,  8.07799062e-01,  1.22274785e+01],
       [ 6.97248902e+00,  8.32340609e-01,  1.22385492e+01],
       [-5.85089513e-02,  3.23689968e+00,  1.22461944e+01],
       [ 2.83536391e+00,  3.23516666e+00,  1.22287271e+01],
       [ 4.27077539e+00,  2.53261964e+00,  1.41553430e+01],
       [ 4.18495777e+00,  2.56448520e+00,  1.49037256e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.68909893e-03,  1.61491699e+00,  8.95337963e+00],
       [ 2.81653372e+00,  1.62847798e+00,  8.98563397e+00],
       [ 5.60187053e+00,  1.62172147e+00,  8.98691111e+00],
       [ 1.40565552e+00,  4.04363073e+00,  8.95726329e+00],
       [ 4.21339287e+00,  4.04562309e+00,  8.98756066e+00],
       [ 7.01194289e+00,  4.04811935e+00,  8.95844112e+00],
       [ 4.27453269e-04,  3.29258461e-03,  1.13134381e+01],
       [ 2.80295990e+00, -8.60655952e-04,  1.13104648e+01],
       [ 5.61906163e+00, -1.81041003e-03,  1.13630866e+01],
       [ 1.39983704e+00,  2.42886556e+00,  1.13551070e+01],
       [ 4.21284948e+00,  2.42643737e+00,  1.15072044e+01],
       [ 7.02157884e+00,  2.42994508e+00,  1.13102993e+01],
       [ 5.59078490e+00,  2.90127471e-02,  1.50550220e+01],
       [ 5.60225892e+00,  1.77102732e-02,  1.40714497e+01],
       [ 6.07580744e+00,  8.69485991e-01,  1.52944131e+01],
       [ 1.36021136e+00,  2.47157468e+00,  1.50897494e+01],
       [ 1.84014573e+00,  3.31707504e+00,  1.53186428e+01],
       [ 1.35469237e+00,  2.45752310e+00,  1.41070323e+01],
       [ 6.93264096e+00,  2.37718230e+00,  1.57748487e+01],
       [ 6.52422533e+00,  3.23200423e+00,  1.54957768e+01],
       [ 7.88352152e+00,  2.44647469e+00,  1.55178262e+01],
       [ 2.72514384e+00, -3.15191384e-02,  1.57915213e+01],
       [ 3.66717522e+00,  2.76518109e-02,  1.55041020e+01],
       [ 2.31433830e+00,  8.26400054e-01,  1.55246837e+01],
       [ 1.38252213e+00,  8.05069948e-01,  1.22359879e+01],
       [ 4.18958966e+00,  5.99375194e-01,  1.22709088e+01],
       [ 7.04796452e+00,  8.24495728e-01,  1.22350838e+01],
       [-1.02272354e-03,  3.24338395e+00,  1.22274856e+01],
       [ 2.62197208e+00,  3.35337219e+00,  1.22681971e+01],
       [ 5.85635325e+00,  3.34659330e+00,  1.22857853e+01],
       [ 4.22781282e+00,  2.43693528e+00,  1.30658735e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 3.05767360e-02,  1.62925051e+00,  8.97198715e+00],
       [ 2.81581259e+00,  1.63163746e+00,  8.98345034e+00],
       [ 5.64263694e+00,  1.62947219e+00,  8.93879956e+00],
       [ 1.42558920e+00,  4.05580366e+00,  8.98981618e+00],
       [ 4.22489447e+00,  4.06899712e+00,  8.96344355e+00],
       [ 7.04625778e+00,  4.06355446e+00,  8.97459038e+00],
       [-1.40332852e-03,  1.71491770e-02,  1.13347406e+01],
       [ 2.84091730e+00,  3.10556424e-03,  1.13880531e+01],
       [ 5.62164236e+00, -7.45457081e-03,  1.13283067e+01],
       [ 1.41670472e+00,  2.45018546e+00,  1.14142698e+01],
       [ 4.25593038e+00,  2.43617311e+00,  1.12650401e+01],
       [ 7.00699250e+00,  2.44828507e+00,  1.12949865e+01],
       [ 5.63944824e+00,  1.45755362e-02,  1.50991693e+01],
       [ 5.67847834e+00, -3.68781601e-03,  1.41162102e+01],
       [ 6.11364910e+00,  8.59985399e-01,  1.53445333e+01],
       [ 1.35376241e+00,  2.45538517e+00,  1.50894598e+01],
       [ 1.83147727e+00,  3.30073843e+00,  1.53175003e+01],
       [ 1.33547686e+00,  2.44770583e+00,  1.41042196e+01],
       [ 6.95332416e+00,  2.36600091e+00,  1.58300091e+01],
       [ 6.54522140e+00,  3.22565047e+00,  1.55637048e+01],
       [ 7.89797949e+00,  2.43294605e+00,  1.55489884e+01],
       [ 2.73168625e+00, -3.27594223e-02,  1.58043235e+01],
       [ 3.67282266e+00,  3.10327096e-02,  1.55181218e+01],
       [ 2.31454744e+00,  8.18464213e-01,  1.55257228e+01],
       [ 1.43410290e+00,  8.64684794e-01,  1.22742686e+01],
       [ 4.27792468e+00,  1.11545137e-01,  1.24192004e+01],
       [ 6.99366796e+00,  8.36758403e-01,  1.22346330e+01],
       [-9.90984035e-02,  3.24638631e+00,  1.22370513e+01],
       [ 2.35056458e+00,  3.52179331e+00,  1.24103080e+01],
       [ 4.85693722e+00,  2.57436816e+00,  1.31760443e+01],
       [ 4.10134964e+00,  2.33855052e+00,  1.32776097e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH9',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-2.33715808e-05,  1.62273921e+00,  8.97172291e+00],
       [ 2.79388368e+00,  1.62471690e+00,  8.96908475e+00],
       [ 5.61070237e+00,  1.62365286e+00,  8.95477085e+00],
       [ 1.39558973e+00,  4.04997137e+00,  8.98107777e+00],
       [ 4.20322579e+00,  4.05970786e+00,  8.96978276e+00],
       [ 7.01548727e+00,  4.05746270e+00,  8.97187858e+00],
       [-2.07097619e-02,  3.24038356e-04,  1.13598714e+01],
       [ 2.81307773e+00,  2.20839984e-03,  1.13466381e+01],
       [ 5.60975232e+00, -2.85155065e-03,  1.13419797e+01],
       [ 1.39340318e+00,  2.43051153e+00,  1.13904948e+01],
       [ 4.23969571e+00,  2.45394342e+00,  1.12948592e+01],
       [ 6.97690642e+00,  2.45615436e+00,  1.13042824e+01],
       [ 5.71499186e+00, -2.91744717e-02,  1.50929362e+01],
       [ 5.67558586e+00, -8.22982662e-02,  1.41128668e+01],
       [ 6.17178223e+00,  8.39113749e-01,  1.52778601e+01],
       [ 1.31908155e+00,  2.51018659e+00,  1.51561059e+01],
       [ 1.79889246e+00,  3.35490964e+00,  1.53846774e+01],
       [ 1.34548108e+00,  2.48292470e+00,  1.41742539e+01],
       [ 6.97621480e+00,  2.34956026e+00,  1.58172595e+01],
       [ 6.55915305e+00,  3.19905369e+00,  1.55286721e+01],
       [ 7.92825282e+00,  2.43523078e+00,  1.55624414e+01],
       [ 2.69149440e+00, -3.64027989e-03,  1.58574001e+01],
       [ 3.61696294e+00,  4.29933646e-02,  1.55295455e+01],
       [ 2.27986255e+00,  8.56118888e-01,  1.56005026e+01],
       [ 1.40006390e+00,  8.08341653e-01,  1.22590215e+01],
       [ 4.23847347e+00,  8.08313655e-01,  1.22282924e+01],
       [ 6.97244499e+00,  8.32617533e-01,  1.22389188e+01],
       [-5.85822339e-02,  3.23684339e+00,  1.22462303e+01],
       [ 2.83580993e+00,  3.23487010e+00,  1.22286579e+01],
       [ 4.27010136e+00,  2.53173778e+00,  1.41550634e+01],
       [ 4.18559032e+00,  2.56266869e+00,  1.49032892e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5]))
}

# Tafel reaction between two top adsorbed hydrogen atoms. Initial,
# transition and final states are reported at each applied potential.
Tafel_top_top = {

'IS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.96385828e-03,  1.65271522e+00,  8.98505635e+00],
       [ 2.80366607e+00,  1.61696390e+00,  8.95939264e+00],
       [ 5.63214882e+00,  1.63659447e+00,  8.97856452e+00],
       [ 1.40936375e+00,  4.05462313e+00,  8.96619001e+00],
       [ 4.21630941e+00,  4.05682266e+00,  8.96260321e+00],
       [ 7.01359605e+00,  4.06232909e+00,  8.98758331e+00],
       [-5.87611840e-03,  3.06855287e-02,  1.12737201e+01],
       [ 2.80967433e+00, -3.67234888e-03,  1.13563186e+01],
       [ 5.58485328e+00,  1.86240374e-02,  1.13651805e+01],
       [ 1.42588115e+00,  2.45170253e+00,  1.13800271e+01],
       [ 4.18374054e+00,  2.39972389e+00,  1.13079152e+01],
       [ 7.02402953e+00,  2.42453693e+00,  1.15028602e+01],
       [ 5.57372900e+00, -3.08872427e-03,  1.50515026e+01],
       [ 5.54685420e+00, -2.76338738e-03,  1.40705885e+01],
       [ 6.06230146e+00,  8.39013479e-01,  1.52704965e+01],
       [ 1.40351048e+00,  2.43367184e+00,  1.50980807e+01],
       [ 1.88171433e+00,  3.27917772e+00,  1.53361515e+01],
       [ 1.42383023e+00,  2.41295075e+00,  1.41176490e+01],
       [ 6.95074697e+00,  2.35214486e+00,  1.56101323e+01],
       [ 6.55379637e+00,  3.24030808e+00,  1.54390149e+01],
       [ 7.92115623e+00,  2.44753086e+00,  1.54534207e+01],
       [ 2.74809304e+00, -7.90873606e-02,  1.58110387e+01],
       [ 3.69451866e+00, -1.13499271e-02,  1.55389615e+01],
       [ 2.34022084e+00,  7.81304891e-01,  1.55505427e+01],
       [ 7.01392923e+00,  2.35647493e+00,  1.30615161e+01],
       [ 1.40342351e+00,  8.64420602e-01,  1.22497850e+01],
       [ 4.17955695e+00,  1.89244051e-01,  1.24206773e+01],
       [ 7.04756702e+00,  5.30821381e-01,  1.22833655e+01],
       [ 8.49249207e-02,  3.33928000e+00,  1.22191308e+01],
       [ 2.33112357e+00,  3.53743195e+00,  1.24074842e+01],
       [ 5.57413092e+00,  3.29683123e+00,  1.21730993e+01],
       [ 4.03232486e+00,  2.27485852e+00,  1.28572240e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.05010312e-02,  1.65136828e+00,  8.95351653e+00],
       [ 2.79447077e+00,  1.61196468e+00,  8.96795470e+00],
       [ 5.61472041e+00,  1.59826593e+00,  8.97792513e+00],
       [ 1.39819568e+00,  4.05720882e+00,  8.96215474e+00],
       [ 4.21144909e+00,  4.03669226e+00,  8.99553427e+00],
       [ 6.98808112e+00,  4.07056168e+00,  8.97268790e+00],
       [-1.33944361e-02,  7.67033277e-02,  1.12486738e+01],
       [ 2.80841812e+00, -3.73821897e-03,  1.13568028e+01],
       [ 5.59364785e+00,  2.84789533e-02,  1.15924682e+01],
       [ 1.41279156e+00,  2.47734606e+00,  1.13791344e+01],
       [ 4.20978526e+00,  2.42447709e+00,  1.12852051e+01],
       [ 7.00765556e+00,  2.51000015e+00,  1.13204077e+01],
       [ 5.72152785e+00, -1.73758690e-01,  1.52874168e+01],
       [ 5.73938271e+00, -1.64360299e-01,  1.43069068e+01],
       [ 6.20916266e+00,  6.62269778e-01,  1.55404694e+01],
       [ 1.46845023e+00,  2.18448017e+00,  1.51549949e+01],
       [ 1.95551543e+00,  3.02509936e+00,  1.53896514e+01],
       [ 1.42604539e+00,  2.19924138e+00,  1.41760926e+01],
       [ 7.08733573e+00,  2.15159666e+00,  1.59794292e+01],
       [ 6.67903565e+00,  3.01458440e+00,  1.57310746e+01],
       [ 8.02777289e+00,  2.20822575e+00,  1.56830033e+01],
       [ 2.81853991e+00, -3.35971890e-01,  1.58341808e+01],
       [ 3.78227873e+00, -2.35211105e-01,  1.56517855e+01],
       [ 2.41019472e+00,  5.28268568e-01,  1.55879296e+01],
       [ 4.74057848e+00,  1.67526454e+00,  1.27255774e+01],
       [ 1.38861744e+00,  8.91872227e-01,  1.22300243e+01],
       [ 4.06466651e+00, -1.08684153e-01,  1.25109664e+01],
       [ 7.10382152e+00,  9.01573459e-01,  1.22534663e+01],
       [-4.23316940e-03,  3.18549066e+00,  1.22112063e+01],
       [ 2.21335951e+00,  3.58573723e+00,  1.24273852e+01],
       [ 6.24373590e+00,  3.53779954e+00,  1.24963826e+01],
       [ 4.09432491e+00,  2.40600437e+00,  1.29522115e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_0VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-8.17261271e-05,  1.61459137e+00,  8.96934314e+00],
       [ 2.80130359e+00,  1.61629693e+00,  8.97017499e+00],
       [ 5.61325516e+00,  1.61283203e+00,  8.96846330e+00],
       [ 1.40077919e+00,  4.04201924e+00,  8.96924934e+00],
       [ 4.20932725e+00,  4.04700376e+00,  8.96801210e+00],
       [ 7.01494845e+00,  4.04691147e+00,  8.96904284e+00],
       [-1.28256182e-03, -1.51062045e-03,  1.13367054e+01],
       [ 2.80454080e+00, -4.60516479e-04,  1.13325574e+01],
       [ 5.61304190e+00, -1.40689563e-03,  1.13635209e+01],
       [ 1.40037772e+00,  2.42912092e+00,  1.13709775e+01],
       [ 4.20797282e+00,  2.42977544e+00,  1.13315745e+01],
       [ 7.01568958e+00,  2.42891567e+00,  1.13338900e+01],
       [ 5.72656546e+00, -2.17329992e-02,  1.51181028e+01],
       [ 5.67296904e+00, -3.26490597e-02,  1.41405125e+01],
       [ 6.18264047e+00,  8.43216388e-01,  1.53296359e+01],
       [ 1.30748194e+00,  2.40239656e+00,  1.50784931e+01],
       [ 1.79131050e+00,  3.24112816e+00,  1.53214763e+01],
       [ 1.32675750e+00,  2.39181137e+00,  1.40974260e+01],
       [ 6.99754418e+00,  2.33842913e+00,  1.58234921e+01],
       [ 6.58899617e+00,  3.19630227e+00,  1.55494517e+01],
       [ 7.94588470e+00,  2.40051744e+00,  1.55455266e+01],
       [ 2.64960074e+00, -1.11347945e-01,  1.58285700e+01],
       [ 3.59532499e+00, -5.02557987e-02,  1.55720379e+01],
       [ 2.24933626e+00,  7.50263870e-01,  1.55603684e+01],
       [ 4.20917735e+00,  2.47169679e+00,  1.51937944e+01],
       [ 1.40567542e+00,  7.96672787e-01,  1.22557839e+01],
       [ 4.19955034e+00,  8.05533471e-01,  1.22461472e+01],
       [ 7.02199703e+00,  8.11211759e-01,  1.22520910e+01],
       [-1.22909625e-02,  3.24592341e+00,  1.22538330e+01],
       [ 2.80792479e+00,  3.24204771e+00,  1.22479504e+01],
       [ 5.61621108e+00,  3.22846263e+00,  1.22451870e+01],
       [ 4.21281641e+00,  2.45056857e+00,  1.44431406e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.93137620e-03,  1.65267842e+00,  8.98477045e+00],
       [ 2.80361300e+00,  1.61703684e+00,  8.95911796e+00],
       [ 5.63212842e+00,  1.63659057e+00,  8.97849483e+00],
       [ 1.40930133e+00,  4.05465249e+00,  8.96600556e+00],
       [ 4.21635452e+00,  4.05679736e+00,  8.96253073e+00],
       [ 7.01356695e+00,  4.06227364e+00,  8.98741211e+00],
       [-5.83892304e-03,  3.06905618e-02,  1.12735568e+01],
       [ 2.80963016e+00, -3.72314136e-03,  1.13563271e+01],
       [ 5.58485063e+00,  1.86006363e-02,  1.13650892e+01],
       [ 1.42580468e+00,  2.45167268e+00,  1.13799372e+01],
       [ 4.18387721e+00,  2.39979339e+00,  1.13077184e+01],
       [ 7.02380646e+00,  2.42450108e+00,  1.15027882e+01],
       [ 5.57379507e+00, -2.87004285e-03,  1.50520491e+01],
       [ 5.54681222e+00, -2.86560321e-03,  1.40699276e+01],
       [ 6.06232106e+00,  8.38958813e-01,  1.52705919e+01],
       [ 1.40353132e+00,  2.43377333e+00,  1.50988393e+01],
       [ 1.88162408e+00,  3.27901380e+00,  1.53360076e+01],
       [ 1.42390592e+00,  2.41300609e+00,  1.41171856e+01],
       [ 6.95069309e+00,  2.35220737e+00,  1.56104378e+01],
       [ 6.55371339e+00,  3.24039711e+00,  1.54389642e+01],
       [ 7.92113084e+00,  2.44760014e+00,  1.54534010e+01],
       [ 2.74798303e+00, -7.92041581e-02,  1.58111239e+01],
       [ 3.69451805e+00, -1.13454553e-02,  1.55389560e+01],
       [ 2.34006350e+00,  7.81338804e-01,  1.55506419e+01],
       [ 7.01400891e+00,  2.35644079e+00,  1.30615530e+01],
       [ 1.40334998e+00,  8.64485469e-01,  1.22496071e+01],
       [ 4.17949275e+00,  1.89221028e-01,  1.24206305e+01],
       [ 7.04760032e+00,  5.30737650e-01,  1.22832859e+01],
       [ 8.50083278e-02,  3.33935635e+00,  1.22190495e+01],
       [ 2.33104293e+00,  3.53748691e+00,  1.24073771e+01],
       [ 5.57414534e+00,  3.29687409e+00,  1.21730421e+01],
       [ 4.03230782e+00,  2.27489116e+00,  1.28571295e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.05010312e-02,  1.65136828e+00,  8.95351653e+00],
       [ 2.79447077e+00,  1.61196468e+00,  8.96795470e+00],
       [ 5.61472041e+00,  1.59826593e+00,  8.97792513e+00],
       [ 1.39819568e+00,  4.05720882e+00,  8.96215474e+00],
       [ 4.21144909e+00,  4.03669226e+00,  8.99553427e+00],
       [ 6.98808112e+00,  4.07056168e+00,  8.97268790e+00],
       [-1.33944361e-02,  7.67033277e-02,  1.12486738e+01],
       [ 2.80841812e+00, -3.73821897e-03,  1.13568028e+01],
       [ 5.59364785e+00,  2.84789533e-02,  1.15924682e+01],
       [ 1.41279156e+00,  2.47734606e+00,  1.13791344e+01],
       [ 4.20978526e+00,  2.42447709e+00,  1.12852051e+01],
       [ 7.00765556e+00,  2.51000015e+00,  1.13204077e+01],
       [ 5.72152785e+00, -1.73758690e-01,  1.52874168e+01],
       [ 5.73938271e+00, -1.64360299e-01,  1.43069068e+01],
       [ 6.20916266e+00,  6.62269778e-01,  1.55404694e+01],
       [ 1.46845023e+00,  2.18448017e+00,  1.51549949e+01],
       [ 1.95551543e+00,  3.02509936e+00,  1.53896514e+01],
       [ 1.42604539e+00,  2.19924138e+00,  1.41760926e+01],
       [ 7.08733573e+00,  2.15159666e+00,  1.59794292e+01],
       [ 6.67903565e+00,  3.01458440e+00,  1.57310746e+01],
       [ 8.02777289e+00,  2.20822575e+00,  1.56830033e+01],
       [ 2.81853991e+00, -3.35971890e-01,  1.58341808e+01],
       [ 3.78227873e+00, -2.35211105e-01,  1.56517855e+01],
       [ 2.41019472e+00,  5.28268568e-01,  1.55879296e+01],
       [ 4.74057848e+00,  1.67526454e+00,  1.27255774e+01],
       [ 1.38861744e+00,  8.91872227e-01,  1.22300243e+01],
       [ 4.06466651e+00, -1.08684153e-01,  1.25109664e+01],
       [ 7.10382152e+00,  9.01573459e-01,  1.22534663e+01],
       [-4.23316940e-03,  3.18549066e+00,  1.22112063e+01],
       [ 2.21335951e+00,  3.58573723e+00,  1.24273852e+01],
       [ 6.24373590e+00,  3.53779954e+00,  1.24963826e+01],
       [ 4.09432491e+00,  2.40600437e+00,  1.29522115e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.2VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-8.10130917e-05,  1.61465981e+00,  8.96937859e+00],
       [ 2.80140419e+00,  1.61628286e+00,  8.96998182e+00],
       [ 5.61315588e+00,  1.61288365e+00,  8.96840999e+00],
       [ 1.40076947e+00,  4.04216522e+00,  8.96929955e+00],
       [ 4.20943807e+00,  4.04708334e+00,  8.96804653e+00],
       [ 7.01493719e+00,  4.04690530e+00,  8.96905062e+00],
       [-1.39676908e-03, -1.49486760e-03,  1.13368840e+01],
       [ 2.80457478e+00, -6.97357616e-04,  1.13327406e+01],
       [ 5.61301601e+00, -1.42877459e-03,  1.13637504e+01],
       [ 1.40024291e+00,  2.42911289e+00,  1.13713335e+01],
       [ 4.20802385e+00,  2.42979120e+00,  1.13316712e+01],
       [ 7.01595321e+00,  2.42892909e+00,  1.13339482e+01],
       [ 5.72657999e+00, -2.15786795e-02,  1.51186015e+01],
       [ 5.67292862e+00, -3.26828815e-02,  1.41400977e+01],
       [ 6.18259554e+00,  8.43100315e-01,  1.53295324e+01],
       [ 1.30743947e+00,  2.40243057e+00,  1.50789642e+01],
       [ 1.79116994e+00,  3.24099688e+00,  1.53215077e+01],
       [ 1.32666667e+00,  2.39176053e+00,  1.40969382e+01],
       [ 6.99752551e+00,  2.33841986e+00,  1.58239097e+01],
       [ 6.58904514e+00,  3.19632570e+00,  1.55493236e+01],
       [ 7.94582401e+00,  2.40057620e+00,  1.55454160e+01],
       [ 2.64955703e+00, -1.11257598e-01,  1.58288835e+01],
       [ 3.59523986e+00, -5.01761712e-02,  1.55718231e+01],
       [ 2.24931830e+00,  7.50427571e-01,  1.55602024e+01],
       [ 4.20919113e+00,  2.47154341e+00,  1.51934532e+01],
       [ 1.40560558e+00,  7.96722585e-01,  1.22556041e+01],
       [ 4.19965199e+00,  8.05473019e-01,  1.22461842e+01],
       [ 7.02202094e+00,  8.11204307e-01,  1.22519766e+01],
       [-1.22579288e-02,  3.24591672e+00,  1.22537317e+01],
       [ 2.80794598e+00,  3.24211922e+00,  1.22479088e+01],
       [ 5.61620376e+00,  3.22857952e+00,  1.22452852e+01],
       [ 4.21272907e+00,  2.45051810e+00,  1.44434727e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.40019744e-03,  1.65189173e+00,  8.98050850e+00],
       [ 2.80272708e+00,  1.61790389e+00,  8.95539700e+00],
       [ 5.63157840e+00,  1.63621269e+00,  8.97760301e+00],
       [ 1.40843566e+00,  4.05497368e+00,  8.96327557e+00],
       [ 4.21671660e+00,  4.05663054e+00,  8.96134880e+00],
       [ 7.01287432e+00,  4.06116835e+00,  8.98486770e+00],
       [-5.88316263e-03,  3.05525348e-02,  1.12706144e+01],
       [ 2.80926523e+00, -4.04428697e-03,  1.13561391e+01],
       [ 5.58487498e+00,  1.86006523e-02,  1.13664925e+01],
       [ 1.42439539e+00,  2.45116641e+00,  1.13808299e+01],
       [ 4.18454882e+00,  2.40007849e+00,  1.13038339e+01],
       [ 7.02135645e+00,  2.42404802e+00,  1.15009740e+01],
       [ 5.57317327e+00, -1.41997422e-03,  1.50497388e+01],
       [ 5.54636345e+00, -4.39882325e-03,  1.40664114e+01],
       [ 6.06328145e+00,  8.39287397e-01,  1.52701654e+01],
       [ 1.40299122e+00,  2.43269911e+00,  1.50984585e+01],
       [ 1.88115652e+00,  3.27808735e+00,  1.53337923e+01],
       [ 1.42442915e+00,  2.41322537e+00,  1.41161367e+01],
       [ 6.95142775e+00,  2.35635324e+00,  1.56234499e+01],
       [ 6.55235996e+00,  3.24065272e+00,  1.54354268e+01],
       [ 7.91929364e+00,  2.44880918e+00,  1.54505371e+01],
       [ 2.74743478e+00, -7.80591209e-02,  1.58158354e+01],
       [ 3.69250570e+00, -1.13624920e-02,  1.55372193e+01],
       [ 2.33745984e+00,  7.80463485e-01,  1.55512374e+01],
       [ 7.01562122e+00,  2.35557639e+00,  1.30597248e+01],
       [ 1.40157973e+00,  8.65669448e-01,  1.22458704e+01],
       [ 4.17813664e+00,  1.88381933e-01,  1.24195452e+01],
       [ 7.04855408e+00,  5.27914738e-01,  1.22808601e+01],
       [ 8.70698227e-02,  3.34153453e+00,  1.22160999e+01],
       [ 2.32909092e+00,  3.53869589e+00,  1.24051631e+01],
       [ 5.57428292e+00,  3.29837240e+00,  1.21708164e+01],
       [ 4.03175022e+00,  2.27553285e+00,  1.28542876e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-1.04976965e-02,  1.65129005e+00,  8.95340996e+00],
       [ 2.79433976e+00,  1.61185690e+00,  8.96777299e+00],
       [ 5.61486198e+00,  1.59831269e+00,  8.97770855e+00],
       [ 1.39813422e+00,  4.05728308e+00,  8.96216902e+00],
       [ 4.21153050e+00,  4.03691198e+00,  8.99503857e+00],
       [ 6.98815056e+00,  4.07046424e+00,  8.97270967e+00],
       [-1.33759332e-02,  7.65997129e-02,  1.12486591e+01],
       [ 2.80834301e+00, -3.77118335e-03,  1.13568225e+01],
       [ 5.59395391e+00,  2.83465568e-02,  1.15927283e+01],
       [ 1.41281345e+00,  2.47738533e+00,  1.13796968e+01],
       [ 4.20973312e+00,  2.42457993e+00,  1.12847771e+01],
       [ 7.00729545e+00,  2.51000180e+00,  1.13208636e+01],
       [ 5.72162237e+00, -1.73670114e-01,  1.52882956e+01],
       [ 5.73935376e+00, -1.64336466e-01,  1.43059189e+01],
       [ 6.20896746e+00,  6.62136537e-01,  1.55403167e+01],
       [ 1.46843909e+00,  2.18453246e+00,  1.51558549e+01],
       [ 1.95547146e+00,  3.02494593e+00,  1.53894207e+01],
       [ 1.42604514e+00,  2.19923372e+00,  1.41750857e+01],
       [ 7.08732272e+00,  2.15170622e+00,  1.59798395e+01],
       [ 6.67896582e+00,  3.01480936e+00,  1.57308305e+01],
       [ 8.02786291e+00,  2.20831154e+00,  1.56827163e+01],
       [ 2.81864853e+00, -3.35916359e-01,  1.58347183e+01],
       [ 3.78235379e+00, -2.35199222e-01,  1.56514953e+01],
       [ 2.41019409e+00,  5.28366132e-01,  1.55876442e+01],
       [ 4.74067407e+00,  1.67529915e+00,  1.27259002e+01],
       [ 1.38856170e+00,  8.91828136e-01,  1.22300704e+01],
       [ 4.06480824e+00, -1.08681626e-01,  1.25109433e+01],
       [ 7.10386576e+00,  9.01537624e-01,  1.22533808e+01],
       [-4.10598790e-03,  3.18563151e+00,  1.22110934e+01],
       [ 2.21315451e+00,  3.58569595e+00,  1.24272406e+01],
       [ 6.24397354e+00,  3.53774326e+00,  1.24960603e+01],
       [ 4.09433968e+00,  2.40630304e+00,  1.29521679e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.4VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.62808636e-04,  1.61560198e+00,  8.96974105e+00],
       [ 2.80223813e+00,  1.61640961e+00,  8.96872314e+00],
       [ 5.61238221e+00,  1.61344539e+00,  8.96825365e+00],
       [ 1.40077056e+00,  4.04364611e+00,  8.97015628e+00],
       [ 4.21056422e+00,  4.04792187e+00,  8.96854526e+00],
       [ 7.01484009e+00,  4.04709329e+00,  8.96920889e+00],
       [-2.25051380e-03, -1.31488502e-03,  1.13387955e+01],
       [ 2.80486896e+00, -2.33710737e-03,  1.13343843e+01],
       [ 5.61284733e+00, -1.71685971e-03,  1.13679802e+01],
       [ 1.39979810e+00,  2.42860744e+00,  1.13769040e+01],
       [ 4.20868158e+00,  2.42935451e+00,  1.13326916e+01],
       [ 7.01743294e+00,  2.42913652e+00,  1.13344160e+01],
       [ 5.72624162e+00, -2.17090330e-02,  1.51184119e+01],
       [ 5.67207504e+00, -3.36220322e-02,  1.41394385e+01],
       [ 6.18225995e+00,  8.43045303e-01,  1.53275302e+01],
       [ 1.30548298e+00,  2.40188601e+00,  1.50783560e+01],
       [ 1.78989834e+00,  3.23950289e+00,  1.53210473e+01],
       [ 1.32536165e+00,  2.39081821e+00,  1.40954453e+01],
       [ 6.99841795e+00,  2.34086616e+00,  1.58304734e+01],
       [ 6.58907369e+00,  3.19536607e+00,  1.55465657e+01],
       [ 7.94449249e+00,  2.40076903e+00,  1.55432977e+01],
       [ 2.65028592e+00, -1.07188733e-01,  1.58345748e+01],
       [ 3.59312552e+00, -4.97724981e-02,  1.55670304e+01],
       [ 2.24944434e+00,  7.51563040e-01,  1.55569346e+01],
       [ 4.20927587e+00,  2.46952439e+00,  1.51937422e+01],
       [ 1.40477538e+00,  7.97309288e-01,  1.22538315e+01],
       [ 4.20093635e+00,  8.04800466e-01,  1.22469035e+01],
       [ 7.02227291e+00,  8.11180877e-01,  1.22509912e+01],
       [-1.18793408e-02,  3.24589672e+00,  1.22531638e+01],
       [ 2.80835183e+00,  3.24294841e+00,  1.22478873e+01],
       [ 5.61614065e+00,  3.23001906e+00,  1.22465814e+01],
       [ 4.21149193e+00,  2.44971896e+00,  1.44435483e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'IS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 1.29971085e-03,  1.65182932e+00,  8.98133481e+00],
       [ 2.80213544e+00,  1.61663608e+00,  8.95668736e+00],
       [ 5.63106058e+00,  1.63513249e+00,  8.97706016e+00],
       [ 1.40872540e+00,  4.05345216e+00,  8.96379086e+00],
       [ 4.21579684e+00,  4.05648564e+00,  8.96086765e+00],
       [ 7.01119456e+00,  4.06024034e+00,  8.98517027e+00],
       [-7.19766404e-03,  3.14907749e-02,  1.12669746e+01],
       [ 2.80862315e+00, -3.78847489e-03,  1.13538309e+01],
       [ 5.58351799e+00,  1.93159410e-02,  1.13685537e+01],
       [ 1.42483279e+00,  2.45176721e+00,  1.13808343e+01],
       [ 4.18354689e+00,  2.39995910e+00,  1.13011961e+01],
       [ 7.02212014e+00,  2.42483250e+00,  1.14962344e+01],
       [ 5.57267567e+00, -1.04962036e-03,  1.50459462e+01],
       [ 5.54603006e+00, -2.38205940e-03,  1.40619336e+01],
       [ 6.06285249e+00,  8.39209011e-01,  1.52657627e+01],
       [ 1.40258154e+00,  2.43212090e+00,  1.50945378e+01],
       [ 1.87990594e+00,  3.27716667e+00,  1.53320897e+01],
       [ 1.42474877e+00,  2.41208399e+00,  1.41114570e+01],
       [ 6.95536852e+00,  2.36115394e+00,  1.56484692e+01],
       [ 6.55178158e+00,  3.23701813e+00,  1.54336807e+01],
       [ 7.91889343e+00,  2.44938017e+00,  1.54485965e+01],
       [ 2.74446077e+00, -7.67104504e-02,  1.58201062e+01],
       [ 3.68773798e+00, -1.15276242e-02,  1.55353502e+01],
       [ 2.33348032e+00,  7.79975108e-01,  1.55506500e+01],
       [ 7.01705925e+00,  2.35378804e+00,  1.30544372e+01],
       [ 1.39913384e+00,  8.64739111e-01,  1.22439477e+01],
       [ 4.17589522e+00,  1.86871020e-01,  1.24202483e+01],
       [ 7.05033480e+00,  5.22420845e-01,  1.22805516e+01],
       [ 8.74702455e-02,  3.34372930e+00,  1.22128934e+01],
       [ 2.32716283e+00,  3.54060183e+00,  1.24049395e+01],
       [ 5.57410464e+00,  3.29796161e+00,  1.21682711e+01],
       [ 4.03064413e+00,  2.27596902e+00,  1.28513646e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'TS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [-9.99965808e-03,  1.64995525e+00,  8.95454693e+00],
       [ 2.79187135e+00,  1.61221588e+00,  8.96565385e+00],
       [ 5.61798718e+00,  1.59968980e+00,  8.97388210e+00],
       [ 1.39714717e+00,  4.05587389e+00,  8.96390669e+00],
       [ 4.21279670e+00,  4.04109764e+00,  8.98492905e+00],
       [ 6.99010751e+00,  4.06730010e+00,  8.97466730e+00],
       [-1.24303207e-02,  7.38276170e-02,  1.12467976e+01],
       [ 2.81057839e+00, -4.02317393e-03,  1.13536226e+01],
       [ 5.59463967e+00,  3.05145130e-02,  1.16002131e+01],
       [ 1.41211089e+00,  2.47477873e+00,  1.13918824e+01],
       [ 4.20919020e+00,  2.42381775e+00,  1.12749306e+01],
       [ 7.00414375e+00,  2.50560187e+00,  1.13254607e+01],
       [ 5.72181208e+00, -1.74943179e-01,  1.52817640e+01],
       [ 5.73644541e+00, -1.64022478e-01,  1.42988581e+01],
       [ 6.20522233e+00,  6.62025337e-01,  1.55353315e+01],
       [ 1.46576101e+00,  2.18403263e+00,  1.51485073e+01],
       [ 1.95500100e+00,  3.02218577e+00,  1.53826242e+01],
       [ 1.42704262e+00,  2.19678673e+00,  1.41668841e+01],
       [ 7.09152268e+00,  2.16060914e+00,  1.59899872e+01],
       [ 6.67893541e+00,  3.01665840e+00,  1.57228862e+01],
       [ 8.02598398e+00,  2.21037676e+00,  1.56733248e+01],
       [ 2.82467474e+00, -3.27057044e-01,  1.58513631e+01],
       [ 3.78360518e+00, -2.35331607e-01,  1.56399903e+01],
       [ 2.41210955e+00,  5.27242895e-01,  1.55777584e+01],
       [ 4.73495470e+00,  1.68644009e+00,  1.27344421e+01],
       [ 1.38654400e+00,  8.88652824e-01,  1.22338860e+01],
       [ 4.06904449e+00, -1.07632478e-01,  1.25104208e+01],
       [ 7.10585927e+00,  9.00818054e-01,  1.22539217e+01],
       [-2.23397107e-03,  3.19282595e+00,  1.22094521e+01],
       [ 2.21012021e+00,  3.58649430e+00,  1.24293872e+01],
       [ 6.24999607e+00,  3.53473287e+00,  1.24965181e+01],
       [ 4.10055805e+00,  2.41269361e+00,  1.29482333e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5])),

'FS_-0.6VSHE':
Atoms(symbols='Pt18OH2OH2OH2OH10',
      positions=[[ 1.40360696e+00,  8.10372857e-01,  6.65200000e+00],
       [ 4.21082088e+00,  8.10372857e-01,  6.65200000e+00],
       [ 7.01803480e+00,  8.10372857e-01,  6.65200000e+00],
       [ 0.00000000e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.80721392e+00,  3.24149143e+00,  6.65200000e+00],
       [ 5.61442784e+00,  3.24149143e+00,  6.65200000e+00],
       [ 2.16277613e-04,  1.61565373e+00,  8.96978777e+00],
       [ 2.80218493e+00,  1.61647747e+00,  8.96890161e+00],
       [ 5.61244856e+00,  1.61344282e+00,  8.96834978e+00],
       [ 1.40078940e+00,  4.04362635e+00,  8.97024484e+00],
       [ 4.21055227e+00,  4.04792581e+00,  8.96859629e+00],
       [ 7.01485234e+00,  4.04715836e+00,  8.96925561e+00],
       [-2.18854488e-03, -1.33488432e-03,  1.13387205e+01],
       [ 2.80488587e+00, -2.20853385e-03,  1.13343003e+01],
       [ 5.61286558e+00, -1.73514246e-03,  1.13681757e+01],
       [ 1.39995227e+00,  2.42858798e+00,  1.13770522e+01],
       [ 4.20868484e+00,  2.42928524e+00,  1.13326700e+01],
       [ 7.01723103e+00,  2.42914869e+00,  1.13343616e+01],
       [ 5.72634792e+00, -2.16284336e-02,  1.51187156e+01],
       [ 5.67195854e+00, -3.37143732e-02,  1.41389325e+01],
       [ 6.18216546e+00,  8.42962194e-01,  1.53273586e+01],
       [ 1.30539676e+00,  2.40193564e+00,  1.50786742e+01],
       [ 1.78981752e+00,  3.23939781e+00,  1.53209145e+01],
       [ 1.32530550e+00,  2.39077652e+00,  1.40949152e+01],
       [ 6.99841771e+00,  2.34092662e+00,  1.58306716e+01],
       [ 6.58891549e+00,  3.19537730e+00,  1.55463648e+01],
       [ 7.94449228e+00,  2.40071259e+00,  1.55431739e+01],
       [ 2.65036470e+00, -1.06916079e-01,  1.58348153e+01],
       [ 3.59314932e+00, -4.97977886e-02,  1.55665988e+01],
       [ 2.24938706e+00,  7.51667813e-01,  1.55566936e+01],
       [ 4.20927314e+00,  2.46939809e+00,  1.51937978e+01],
       [ 1.40474972e+00,  7.97315259e-01,  1.22539203e+01],
       [ 4.20098439e+00,  8.04785130e-01,  1.22470102e+01],
       [ 7.02228119e+00,  8.11194765e-01,  1.22510781e+01],
       [-1.18704074e-02,  3.24590427e+00,  1.22532894e+01],
       [ 2.80839235e+00,  3.24297035e+00,  1.22480047e+01],
       [ 5.61614887e+00,  3.23006328e+00,  1.22466628e+01],
       [ 4.21140557e+00,  2.44967707e+00,  1.44435234e+01]],
      pbc=[True, True, False], cell=[8.421641763931781, 4.862237139424608, 30.0],
      constraint=FixAtoms(indices=[0, 1, 2, 3, 4, 5]))
}

