from ase import Atoms
from ase.build import fcc111, add_adsorbate
from ase.constraints import FixAtoms
import ase.visualize


# define function that returns a Pt(111) slab with adsorbates on top.
def build_slab_with_adsorbate(N=0, Nx=2, Ny=2, l=3):
    """ returns a Pt(111) slab with adsorbates on top.
    Input:
    - N: number of adsorbates
    - Nx: number of surface atoms in first lattice direction
    - Ny: number of surface atoms in second lattice direction
    - l: number of layers
    Output:
    - ase geometry
    """
    # construct 111 slab
    # - first argument: atom type
    # - since no bulk lattice constant is specified, default will be used
    # - l layers, Nx x Ny atoms in the surface unit cell
    # - 3A of vacuum on each side of the cell
    slab = fcc111('Pt', (Nx,Ny,l), vacuum=3.0)

    #add adsorbates
    for i_ads in range(N):
        #find the correct position for the adsorbate (counting atoms in the unit cell)
        i = int(i_ads / Nx) #rows
        j = i_ads % Nx #columns

        # add adsorbate to the slab
        # - third argument: height of adsorbate from uppermost Pt layer
        # - fourth argument: position specifier
        # - place atom in 'fcc' position
        # - offset Hs to place multiple H atoms
        add_adsorbate(slab, 'H', 1, 'fcc', offset = (i,j))

    #freeze some atoms such that they cannot move during geometr optimization
    fixed = []
    for i in range(Nx*Ny*2): #fix 2 layers
        fixed.append(i)
    slab.set_constraint(FixAtoms(indices=fixed))
    slab.pbc[2]=True # make 3D periodic (by default, slabs are treated 2D periodically)

    #retrun the geometry
    return slab

# define function that returns H2 molecule in a box
def build_H2():
    """ returns the geometry of an H2 moleucle in a box."""
    box= 10.0 # box size in Angstrom
    halfbox = box/2
    d = 0.7 # initial H-H distance in Angstrom
    H2 = Atoms('H2',
                 positions=([halfbox - d / 2, halfbox, halfbox],  #correct!!!!
                            [halfbox + d / 2, halfbox, halfbox]),
                 cell=(box, box, box))
    #return the geometry
    return H2


if __name__ == "__main__":
    slab = build_slab_with_adsorbate(3, 2, 2, 3)
    ase.visualize.view(slab)
    h2 = build_H2()
    ase.visualize.view(h2)
